CREATE TABLE belegtyp(
  belt_typ                      VARCHAR(3) NOT NULL PRIMARY KEY,  -- Kürzel mit was für einem Beleg wir es zu tun haben
  belt_bezeichnung              VARCHAR(100),                     -- Bezeichnung der Belegart
  belt_multipledok              BOOLEAN NOT NULL DEFAULT FALSE,   /* Bestimmt ob jedem Beleg genau ein Dokument zugeordnet ist (Eingangsrechnung)
        oder es mehrere Dokumente (Später Einkauf/Verkauf) geben kann. Gibt es mehrere Dokumente, könnnen Adressen pro Position festgelegt werden.
        Gibt es nur ein Dokument, gelten die Adressen aus dem Belegdokument für alle Positionen.
        [OBSOLET?]
        */
  belt_eingangsseitig           BOOLEAN NOT NULL DEFAULT FALSE,   -- Anfrage, Einkauf/Wareneingang/Eingangsrechnung und sowas ... darüber wird entschieden welche Kontierung, Steuercode etc...
  belt_ausgangsseitig           BOOLEAN NOT NULL DEFAULT FALSE,   -- Auftrag, Lieferschein, Rechnung ...
  belt_zahlfaehig               BOOLEAN NOT NULL DEFAULT FALSE,   -- Dieser Beleg erlaubt Zahlungen.
  belt_preiscalc                BOOLEAN NOT NULL DEFAULT FALSE
 );


CREATE TABLE belegart(
  bela_id                SERIAL PRIMARY KEY,
  bela_beld_typ          VARCHAR(3) NOT NULL REFERENCES belegtyp ON UPDATE CASCADE,
  bela_bez               VARCHAR(50)
 );

  CREATE OR REPLACE FUNCTION belegart__a_iud() RETURNS TRIGGER AS $$
    BEGIN
      IF NOT tg_op='INSERT' THEN
        DELETE FROM belegartlang WHERE belal_bela_id=old.bela_id AND belal_spr_key=prodat_languages.current_lang();
      END IF;
      IF NOT tg_op='DELETE' THEN
        INSERT INTO belegartlang (belal_bela_id, belal_spr_key, belal_txt) VALUES (new.bela_id, prodat_languages.current_lang(), new.bela_bez);
      END IF;
      RETURN new;
    END $$ LANGUAGE plpgsql;

   CREATE TRIGGER belegart__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belegart
    FOR EACH ROW
    EXECUTE PROCEDURE belegart__a_iud();


CREATE TABLE belegartlang
 (belal_id              SERIAL PRIMARY KEY,
  belal_bela_id         INTEGER NOT NULL REFERENCES belegart ON UPDATE CASCADE ON DELETE CASCADE,
  belal_spr_key         VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
  belal_txt             VARCHAR(50)
 );

CREATE UNIQUE INDEX belegartlang_unique ON belegartlang(belal_bela_id, belal_spr_key);


-- Belegdokumente. Enthalten Positionen eines oder mehrerer Belege. In belegposition steht Referenz auf Dokument, falls es eines gibt.
CREATE TABLE belegdokument(
  --Grundfelder
  beld_id                       SERIAL NOT NULL PRIMARY KEY,      -- Eindeutiger Schlüssel
  beld_belegtyp                 VARCHAR(3) NOT NULL,              -- Belegtyp den wir auf das Dokument packen (Eingangsrechnung, Lieferschein ... )
  beld_belegart                 INTEGER NOT NULL REFERENCES belegart, --Belegart = Genauerer Belegtyp .... "Gutschrift", "Retourelieferung", Beistell-Lieferschein etc...
  beld_dokunr                   VARCHAR(30) NOT NULL,             -- Dokumentnummer zum Beleg
  beld_titel                    VARCHAR(200),                     -- Belegtitel
  beld_kopftext                 TEXT,                             -- Kopftext d. Dokuments
  beld_kopftext_rtf             TEXT,
  beld_fusstext                 TEXT,                             -- Fußtext d. Dokuments
  beld_fusstext_rtf             TEXT,
  beld_allg1                    VARCHAR(40),                      -- Freies Textfeld 1
  beld_allg2                    VARCHAR(100),                     -- Freies Textfeld 2
  beld_krzbesteller             VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --Dokumentadresse, bspw. Lieferant im Einkauf oder Lieferort bei Eingangsrechnung
  beld_krzlieferung             VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --Dokumentadresse, bspw. Lieferadresse im Einkauf
  beld_krzrechnung              VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --Dokumentadresse, bspw. Rechnungsadresse im Einkauf oder Rechnungssteller bei Eingangsrechnung
  beld_apkrzl                   VARCHAR(10),                      -- Ansprechpartner d. Kunde / Kürzel
  beld_ap                       VARCHAR(100),                     -- Ansprechpartner Kunde / Freitext
  beld_apint                    VARCHAR(10) REFERENCES llv(ll_db_usename) ON UPDATE CASCADE DEFAULT tsystem.current_user_ll_db_usename(), -- Ansprechpartner Intern...Standard angemeldeter Nutzer

  --Statusinformationen
  beld_definitiv                BOOL NOT NULL DEFAULT FALSE,      -- Dokument ist definitiv
  beld_freigabe                 BOOL NOT NULL DEFAULT FALSE,      -- Dokument ist zur Weiterbearbeitung o. Zahlung freigegeben
  beld_verbucht                 BOOL NOT NULL DEFAULT FALSE,      -- Das Dokument ist verbucht, bzw. alle Positionen davon übernommen oder verbucht
  beld_print                    BOOLEAN NOT NULL DEFAULT FALSE,   -- Druckstatus. Das Dokument wurde bereits ausgegeben. (Derzeit Manuell gesetzt)
  beld_bindefrist               VARCHAR(75),                      -- Bindefrist / Gültigkeit des Angebots / Anfrage etc.. (siehe auch beld_bestaetigung_bis)
  beld_erstelldatum             DATE DEFAULT current_date,        -- Erfassungsdatum o. Eingangsdatum des Beleg
  beld_bestaetigung_bis         DATE,                             -- Zeitpunkt, bis zu dem der Empfänger die Angaben bestätigen, bzw. Auftrag abschließen soll
  beld_abschlussdatum           DATE,                             -- Da wurde Beleg abgeschlossen, bezahlt, verbucht oder so
  beld_exportdatum              DATE,                             -- Zeitpunkt des Rechnungsexports (EDI)
  beld_priorisieren             BOOLEAN NOT NULL DEFAULT FALSE,   -- Flag "Wichtig", bevorzugt behandeln gegenüber den anderen Belegen
  --

  -- Zahlungsrelevante Felder
  beld_fixwert                  NUMERIC(16,4),                    -- Fixe Dokumentsumme
  beld_zak                      INTEGER,                          -- Zahlungskondition
  beld_skv                      INTEGER,                          -- Skontoverfall in Tagen ab Belegdatum
  beld_sks                      NUMERIC(5,2),                     -- Skontosatz
  beld_zakbem                   VARCHAR(40),                      -- Zahlungskond. Bemerkung
  beld_netto                    NUMERIC(14,2) NOT NULL DEFAULT 0, -- Nettosumme  über Dokument
  beld_brutto                   NUMERIC(14,2) NOT NULL DEFAULT 0, -- Bruttosumme über Dokument
  beld_netto_basis_w            NUMERIC(14,2) NOT NULL DEFAULT 0, -- Nettosumme  über Dokument
  beld_brutto_basis_w           NUMERIC(14,2) NOT NULL DEFAULT 0, -- Bruttosumme über Dokument
  beld_steuer                   NUMERIC(14,2) NOT NULL DEFAULT 0, -- Steuersumme über Dokument
  beld_skonto                   NUMERIC(14,2) NOT NULL DEFAULT 0, -- Skontobetrag, (ohne Berücksichtigung des Zahlungsdatums)
  beld_bezahlt                  NUMERIC(14,2) NOT NULL DEFAULT 0, -- Bereits bezahlter Betrag
  beld_gesamtrabatt             NUMERIC(5,2)  NOT NULL DEFAULT 0, -- Gesamtrabatt über alle Positionen
  beld_valutadatum              DATE,                             -- Zeitpunkt der Wertstellung (beld_abschlussdatum, wenn wenn leer)
  beld_waehr                    VARCHAR(3) NOT NULL CONSTRAINT xtt4056 REFERENCES bewa DEFAULT TSystem.Settings__Get('BASIS_W'),
  beld_zahlart                  INTEGER REFERENCES zahlart,       --Zahlarten
  beld_gewicht                  NUMERIC(12,4),                    --Gewicht aller Positionen des Dokuments

  -- Versandinformationen
  beld_versandart               INTEGER REFERENCES versart,       -- Versandart
  beld_versandartbem            VARCHAR(100),                     -- Versandart / Freitext
  beld_refbeleg                 VARCHAR(40),                      -- Rechnungsnummer des Lieferanten oder andere Referenz auf Lieferant/Kunde
  beld_versandort               VARCHAR(30) REFERENCES adressen_keys ON UPDATE CASCADE,
  -- System (tables__generate_missing_fields)
  dbrid                         VARCHAR(32) NOT NULL DEFAULT nextval('db_id_seq'),
  insert_date                   DATE,
  insert_by                     VARCHAR(32),
  modified_by                   VARCHAR(32),
  modified_date                 TIMESTAMP(0)
 );


 -- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
 CREATE TRIGGER belegdokument_set_modified
  BEFORE INSERT OR UPDATE
  ON belegdokument
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();

 -- Inizes: Dokument existiert bereits.
 CREATE UNIQUE INDEX xtt10849 ON belegdokument (beld_belegtyp, beld_dokunr);
  CREATE INDEX belegdokument_beld_dokunr ON belegdokument(beld_dokunr);
  CREATE INDEX belegdokument_beld_dokunr_like ON belegdokument(beld_dokunr varchar_pattern_ops);
  CREATE INDEX belegdokument_beld_refbeleg ON belegdokument(beld_refbeleg);
  CREATE INDEX belegdokument_beld_refbeleg_like ON belegdokument(beld_refbeleg varchar_pattern_ops);
  CREATE INDEX belegdokument_insert_date ON belegdokument(insert_date);
  CREATE INDEX belegdokument_beld_erstelldatum ON belegdokument(beld_erstelldatum);
  CREATE INDEX belegdokument_beld_erstelldatum_ym ON belegdokument(date_to_yearmonth(beld_erstelldatum));

 CREATE TABLE belegdokument_keys(
  beld_key      INTEGER NOT NULL PRIMARY KEY
  );


 --Foreign-Keys in Positionen etc. können nicht auf die abgeleiteten Tabellen referenzieren. Wir brauchen also eine extra-Tabelle für die Beleg-Schlüssel.
 CREATE OR REPLACE FUNCTION register_beleg_id() RETURNS TRIGGER AS $$
    BEGIN
     IF tg_op='DELETE' THEN
            DELETE FROM belegdokument_keys WHERE beld_key=old.beld_id;
            RETURN old;
     ELSIF tg_op='UPDATE' THEN
            IF new.beld_id<>old.beld_id THEN
                    UPDATE belegdokument_keys SET beld_key=new.beld_id WHERE beld_key=old.beld_id;
            END IF;
            RETURN new;
     ELSE
            INSERT INTO belegdokument_keys(beld_key) VALUES (new.beld_id);
            RETURN new;
     END IF;
    END $$ LANGUAGE plpgsql;

 CREATE TRIGGER register_beleg_id
    AFTER INSERT OR UPDATE OR DELETE
    ON belegdokument
    FOR EACH ROW
    EXECUTE PROCEDURE register_beleg_id();

--
 CREATE OR REPLACE FUNCTION belegdokument__b_iu() RETURNS TRIGGER AS $$
    DECLARE belegtypRec     RECORD;
            flag            BOOLEAN;
            krz             VARCHAR;
            spco            VARCHAR;
            ident_kopf  VARCHAR := '';
            ident_fuss  VARCHAR := '';
    BEGIN

      -- Trigger 'deaktiviert' also raus.
      IF BelDokTriggerOff() THEN RETURN new; END IF;

      --Belegnummern-Prüfung
      IF (COALESCE(new.beld_dokunr,'') = '') THEN RAISE EXCEPTION '%',lang_text(10984); END IF;

      SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = new.beld_belegtyp;

      new.beld_zak := COALESCE(new.beld_zak,0);
      new.beld_sks := COALESCE(new.beld_sks,0);
      new.beld_skv := COALESCE(new.beld_skv,0);

      -- Nachsetzen von Adressdaten (von oben nach unten übernehmen)
      new.beld_krzbesteller := COALESCE(new.beld_krzbesteller, new.beld_krzlieferung, new.beld_krzrechnung);
      new.beld_krzlieferung := COALESCE(new.beld_krzlieferung, new.beld_krzbesteller);
      new.beld_krzrechnung  := COALESCE(new.beld_krzrechnung,  new.beld_krzlieferung);

      -- Eingangsrechnung
      IF new.beld_belegtyp = 'ERG' THEN
        new.beld_belegart:=COALESCE(new.beld_belegart, 1);
      -- Lieferschein
      ELSIF new.beld_belegtyp = 'LFS' THEN
        new.beld_belegart:=COALESCE(new.beld_belegart, 3);
      -- Lieferantenursprungserklärung
      ELSIF (new.beld_belegtyp = 'LUE') AND (new.beld_belegart IS NULL) THEN
        new.beld_belegart:=bela_id FROM belegart WHERE bela_beld_typ = new.beld_belegtyp;
      END IF;

      SELECT ad_krz INTO krz FROM adressen_view WHERE ad_krz=new.beld_krzlieferung;

      -- Allgemeine und belegspezifische Vorgaben bei Neuanlage
      IF (tg_op='INSERT') THEN

        -- Für alle Belege Währung (auch wenn in LFS z.Bsp. nicht benutzt)
        new.beld_waehr:=COALESCE(new.beld_waehr, TSystem.Settings__Get('BASIS_W'));

        IF new.beld_belegtyp = 'LFS' THEN    -- Lieferschein

          -- Ansprechpartner = Ersteller, falls leer
          new.beld_apint:= COALESCE(new.beld_apint, current_user_ll_db_usename());
          IF new.beld_belegart <> 6 THEN -- 'Normaler' Lieferschein o. Retourelieferung
            ident_kopf := 'DOKLIEFSCH_TXT_KOPF';
            ident_fuss := 'DOKLIEFSCH_TXT_FUSS';
            spco:=prodat_languages.adk1_spco(krz);
          ELSE                          -- Sondertext für Beistell-Lieferschein (#9399)
            ident_kopf := 'DOKLIEFSCHBEISTELL_TXT_KOPF';
            ident_fuss := 'DOKLIEFSCHBEISTELL_TXT_FUSS';
            spco:=prodat_languages.adk1_spco(krz);
          END IF;

        ELSIF new.beld_belegtyp = 'LUE' THEN -- Lieferantenursprungserklärung
          ident_kopf := 'DOKLUE_TXT_KOPF';
          ident_fuss := 'DOKLUE_TXT_FUSS';
          spco:=prodat_languages.adk2_spco(krz);
        END IF;

        -- Anhand von Text-Ident und Sprachcode, die Standardtexte zuweisen
        IF new.beld_kopftext IS NULL AND ident_kopf IS NOT NULL THEN --kopf
          SELECT txt, txtrtf INTO new.beld_kopftext, new.beld_kopftext_rtf FROM belarzu__zu_tit__gettxtrtf(ident_kopf, spco);
        END IF;

        IF new.beld_fusstext IS NULL AND ident_fuss IS NOT NULL THEN --fuß
          SELECT txt, txtrtf INTO new.beld_fusstext, new.beld_fusstext_rtf FROM belarzu__zu_tit__gettxtrtf(ident_fuss, spco);
        END IF;

      END IF; -- END IF (tg_op='INSERT')

      --Normaler Lieferschein
      IF (tg_op = 'UPDATE') AND (new.beld_belegtyp = 'LFS') THEN
        /*Wenn sich Fakturiert-Status (beld_verbucht) im Lieferschein ändert, setzen wir den definitiv Status ggf. um.
          Sind alle Lagerabgänge nicht verrechenbar (z.Bsp. Konsignationslager), wird der Lieferschein autom. fakturiert gesetzt,
          damit er aus Assistenten raus und nicht "offen" ist. Wenn es keine einzige Rechnung auf den Lieferschein gab
          und der trotzdem fakturiert ist, darf er nicht definitiv gesetzt werden, da wir wahrscheinlich noch Sachen anhängen wolle. */
        IF (new.beld_verbucht) AND (NOT new.beld_definitiv) THEN
          SELECT EXISTS (
                  SELECT true FROM belegpos JOIN belzeil_auftg_lif ON bz_belp_id = belp_id
                          WHERE belp_dokument_id = new.beld_id ) INTO flag;
          new.beld_definitiv := flag;
        END IF;
        --Wenn definitiv Status gesetzt wurde, pruefen wir, ob es überhaupt verrechenbare Positionen gibt. Falls nein, fakturiert setzen.
        IF (new.beld_definitiv) AND (NOT old.beld_definitiv) THEN
          SELECT NOT EXISTS(SELECT true FROM belegpos WHERE belp_dokument_id = new.beld_id AND NOT belp_erledigt AND NOT belp_storniert) INTO flag;
          new.beld_verbucht:=flag;
        END IF;
      END IF; -- END IF (tg_op='UPDATE')

      RETURN new;
    END $$ LANGUAGE plpgsql;
--
   CREATE TRIGGER belegdokument__b_iu
    BEFORE INSERT OR UPDATE
    ON belegdokument
    FOR EACH ROW
    EXECUTE PROCEDURE belegdokument__b_iu();


 CREATE OR REPLACE FUNCTION belegdokument__a_i() RETURNS TRIGGER AS $$
    DECLARE belegtypRec RECORD;
    BEGIN

      IF BelDokTriggerOff() THEN -- Wir machen mal lieber nichts.
            RETURN new;
      END IF;

      new.beld_dokunr = UPPER(new.beld_dokunr);
      -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen.
      SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = new.beld_belegtyp;

      -- Belegtyp lässt nur eine 1 zu 1 Abbildung zu Dokumenten zu. Also können wir einfach zu dem neu angelegten Dokument im Hintergrund einen Beleg mit Belegnummer=Dokumentnummer anlegen.
      IF NOT belegtypRec.belt_multipledok THEN
        INSERT INTO beleg(bel_belegtyp, bel_nummer) VALUES (new.beld_belegtyp, new.beld_dokunr);
      END IF;
      RETURN new;
    END $$ LANGUAGE plpgsql;


   CREATE TRIGGER belegdokument__a_i
    AFTER INSERT
    ON belegdokument
    FOR EACH ROW
    EXECUTE PROCEDURE belegdokument__a_i();


 CREATE OR REPLACE FUNCTION belegdokument__a_u() RETURNS TRIGGER AS $$
    DECLARE belegtypRec RECORD;
            oldkurs     NUMERIC;
            newkurs     NUMERIC;
    BEGIN

      IF BelDokTriggerOff() THEN -- Wir machen mal lieber nichts.
        IF Current_User='root' THEN RAISE NOTICE 'Belegdokument.Triggers=OFF.Beld_ID: % Beld_Dokunr: % ', new.beld_id, new.beld_dokunr ; END IF;
        RETURN new;
      END IF;
      IF Current_User='root' THEN RAISE NOTICE 'Belegdokument__a_u Beld_ID: % Beld_Dokunr: % ', new.beld_id, new.beld_dokunr ; END IF;

      -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen.
      SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = new.beld_belegtyp;

      -- Belegtyp lässt nur eine 1 zu 1 Abbildung zu Dokumenten zu. Also können wir einfach zu dem neu angelegten Dokument im Hintergrund einen Beleg mit Belegnummer=Dokumentnummer anlegen.
      IF NOT belegtypRec.belt_multipledok THEN

          --Adressen im Kopf und den Positionen Synchron halten
          IF (new.beld_belegtyp = 'ERG') AND (
             (old.beld_krzbesteller <> new.beld_krzbesteller) OR (old.beld_krzlieferung <> new.beld_krzlieferung) OR (old.beld_krzrechnung <> new.beld_krzrechnung)
             OR (old.beld_erstelldatum <> new.beld_erstelldatum)
             OR (old.beld_waehr <> new.beld_waehr))
            THEN
            PERFORM DisableBelDokTrigger();
            PERFORM DisableBelAbzuTrigger();
            PERFORM DisableBelPosAbzuTrigger();


            IF (old.beld_waehr <> new.beld_waehr) THEN

              --Aktueller Kurs der neuen Währung. Belegpositionen halten den genutzten Kurs. Ab- und Zuschlaege aber nicht. Also muessen wir die anders umrechnen.
              SELECT COALESCE(wa_kurs,0)   INTO newkurs FROM bewa WHERE wa_einh = new.beld_waehr LIMIT 1;

              --Fuer Ab- Zuschlaege nehmen wir den Kurs der letzten Belegposition (das machen wir bei Dokumentsummen auch)
              SELECT COALESCE(belp_kurs,0) INTO oldkurs FROM belegpos WHERE belp_dokument_id = new.beld_id ORDER BY belp_pos DESC LIMIT 1;

              -- Falls da was nicht geklappt hat, nehmen wir den aktuellen Kurs der alten Waehrung.
              IF ( oldkurs = 0 ) THEN
                SELECT COALESCE(wa_kurs,0) INTO oldkurs FROM bewa WHERE wa_einh = old.beld_waehr LIMIT 1;
              END IF;

              --Betraege fuer Belegab- und Zuschlaege neu schreiben.
              UPDATE belegabzu SET  belaz_betrag = belaz_betrag * ( oldkurs / do1If0(newkurs)) WHERE belaz_dokument_id = new.beld_id;
            END IF;

            UPDATE belegpos SET         belp_krzbesteller = new.beld_krzbesteller,
                                        belp_krzlieferung = new.beld_krzlieferung,
                                        belp_krzrechnung = new.beld_krzrechnung,
                                        belp_erstelldatum = new.beld_erstelldatum,
                                        belp_waehr = new.beld_waehr,
                                        belp_kurs = (SELECT COALESCE(wa_kurs,0) FROM bewa WHERE wa_einh = new.beld_waehr LIMIT 1)
                                    WHERE belp_dokument_id = new.beld_id;

            --Summen neu ausrechnen.
            PERFORM TBeleg.Update_DokumentSumme(new.beld_id);

            PERFORM EnableBelDokTrigger();
            PERFORM EnableBelAbzuTrigger();
            PERFORM EnableBelPosAbzuTrigger();
          END IF;

          --Adressen im Kopf aendern, Positionen umschreiben, wo die Adresse vorkam.(Bei Lieferschein wegen einer Lieferung und unterschiedlichen Rechnungsempfaengern)
          IF (new.beld_belegtyp = 'LFS') AND (
             (old.beld_krzbesteller <> new.beld_krzbesteller) OR (old.beld_krzlieferung <> new.beld_krzlieferung) OR (old.beld_krzrechnung <> new.beld_krzrechnung)
             OR (old.beld_erstelldatum <> new.beld_erstelldatum) )
             THEN
             UPDATE belegpos SET        belp_krzbesteller = new.beld_krzbesteller,
                                        belp_krzlieferung = new.beld_krzlieferung,
                                        belp_krzrechnung =  IfThen( belp_krzrechnung = old.beld_krzrechnung, new.beld_krzrechnung, belp_krzrechnung), --Nur dort Rechnungsadresse umschreiben,
                                        --wo Rechnungsaddr. der Position mit der Kopfadresse uebereingestimmte hat.
                                        belp_erstelldatum = new.beld_erstelldatum,
                                        belp_waehr = new.beld_waehr,
                                        belp_kurs = (SELECT COALESCE(wa_kurs,0) FROM bewa WHERE wa_einh = new.beld_waehr LIMIT 1)
                                    WHERE belp_dokument_id = new.beld_id;
          END IF;
      END IF;

      IF(new.beld_print <> old.beld_print) THEN -- Wenn das Setting da ist, wird das Warenausgangsdatum beim Setzen des Print Flags mit gesetzt
        IF TSystem.Settings__GetBool('OnLiefschPrint_SetBelpTermin') THEN
          PERFORM disableBelPosTrigger();
          UPDATE belegpos SET belp_termin = IfThen(new.beld_print,current_date,null)
                    WHERE belp_dokument_id = new.beld_id
                            AND belp_belegtyp = 'LFS'
                            AND IfThen(new.beld_print,belp_termin IS NULL,belp_termin IS NOT NULL) ;
          PERFORM EnableBelPosTrigger();
        END IF;
      END IF;

      --Gesamtsummen aktualisieren bei Rabattänderung
      IF belegtypRec.belt_preiscalc THEN
        IF (new.beld_sks <> old.beld_sks) OR (new.beld_gesamtrabatt <> old.beld_gesamtrabatt) OR (new.beld_skv <> old.beld_skv) THEN
          PERFORM TBeleg.Update_DokumentSumme(new.beld_id);
        END IF;
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

   CREATE TRIGGER belegdokument__a_u
    AFTER UPDATE
    ON belegdokument
    FOR EACH ROW
    EXECUTE PROCEDURE belegdokument__a_u();


 -- Automatisch bei Lieferscheinen die Eigenschaft "Gelangensbestätigung: Unterschrift erhalten" ja/nein anlegen, wenn Gedruckt-Status gesetzt wird. #4512
 CREATE OR REPLACE FUNCTION belegdokument__a_u__lfs_beld_print() RETURNS TRIGGER AS $$
    BEGIN
      -- Der Kunde ist EU-Exporteur
      IF (EXISTS(SELECT TRUE FROM adk1 JOIN adressen_view ON a1_krz=adk_ad_krz WHERE ad_krz=new.beld_krzrechnung AND a1_euexport)) THEN
        PERFORM TRecnoParam.Set('System.Gelangensbestätigung', new.dbrid, False);
      END IF;
      RETURN new;
    END $$ LANGUAGE plpgsql;

   CREATE TRIGGER belegdokument__a_u__lfs_beld_print
    AFTER UPDATE OF beld_print
    ON belegdokument
    FOR EACH ROW
    -- Nur Lieferscheine und nur wenn Gedruckt von falsch auf wahr gesetzt wird, inklusive Setting
    WHEN (new.beld_belegtyp='LFS' AND new.beld_print AND NOT old.beld_print AND TSystem.Settings__GetBool__cascade_save('LFSAutoRecnoGelangensbestaetigung'))
    EXECUTE PROCEDURE belegdokument__a_u__lfs_beld_print();


 CREATE OR REPLACE FUNCTION belegdokument__a_d() RETURNS TRIGGER AS $$
    DECLARE belegtypRec RECORD;
    BEGIN
      IF BelDokTriggerOff() THEN -- Wir machen mal lieber nichts.
          RETURN new;
      END IF;

      -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen.
      SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = old.beld_belegtyp;

      -- Belegtyp lässt nur eine 1 zu 1 Abbildung zu Dokumenten zu. Also können wir einfach den Beleg löschen, wenn das Dokument gelöscht wird.
      IF NOT belegtypRec.belt_multipledok THEN
        DELETE FROM beleg WHERE bel_belegtyp = old.beld_belegtyp AND bel_nummer = old.beld_dokunr;
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

   CREATE TRIGGER belegdokument__a_d
    AFTER DELETE
    ON belegdokument
    FOR EACH ROW
    EXECUTE PROCEDURE belegdokument__a_d();



-- Belege
CREATE TABLE beleg(
  bel_id                        SERIAL NOT NULL PRIMARY KEY,      -- Eindeutiger Schlüssel des Belegs
  bel_belegtyp                  VARCHAR(3) NOT NULL REFERENCES belegtyp,  -- Belegtyp den wir auf das Dokument packen
  bel_nummer                    VARCHAR(30) NOT NULL,             -- Belegnummer aus Oberfläche. Bsp: B2010-0001
  --bel_netto                   NUMERIC(14,2) NOT NULL DEFAULT 0, -- Nettosumme  über Gesamtbeleg
  --bel_brutto                  NUMERIC(14,2) NOT NULL DEFAULT 0, -- Bruttosumme über Gesamtbeleg
  --bel_steuer                  NUMERIC(14,2) NOT NULL DEFAULT 0, -- Steuersumme über Gesamtbeleg
  bel_txt                       TEXT,                             -- Freier Text zum Beleg
  bel_txt_rtf                   TEXT
 );

 --Beleg existiert bereits.
 CREATE UNIQUE INDEX xtt10848 ON beleg (bel_belegtyp, bel_nummer);


-- Ab- / Zuschläge Dokumentbezogen
CREATE TABLE belegabzu (                                                                                    -- VIEWFELDER
  belaz_id            SERIAL NOT NULL PRIMARY KEY,                                                          --  ID des Zuschlags
  belaz_type          VARCHAR(1) DEFAULT 'E',                                                               --  Zuschlagstyp, E-Einmalig, P-Position, M-Per ME
  belaz_pos           INTEGER,                                                                              --  Position
  belaz_abzu_id       INTEGER NOT NULL REFERENCES abzu  ON UPDATE CASCADE,                                  --  ID der Vorgabe
  belaz_dokument_id   INTEGER NOT NULL REFERENCES belegdokument_keys ON UPDATE CASCADE ON DELETE CASCADE,   --  ID des Datensatzes an dem der Zuschlag hängt
  belaz_anzahl        NUMERIC(12,4) NOT NULL DEFAULT 1,                                                     --  Anzahl / Menge
  belaz_betrag        NUMERIC(12,4) NOT NULL DEFAULT 0,                                                     --  Betrag in Währung des Parent-Datensatzes
  belaz_prozent       NUMERIC(12,4) DEFAULT 0,                                                              --  Prozentualer Zuschlag auf Basis des Parent-Betrags
  belaz_canskonto     BOOLEAN NOT NULL DEFAULT TRUE,                                                        --  Gilt Skonto für den Zuschlag?
  belaz_steucode      INTEGER REFERENCES steutxt ON UPDATE CASCADE,                                         --  Steuercode (meist gleich Parent)
  belaz_steuproz      NUMERIC(5,2) NOT NULL DEFAULT 0,                                                      --  Prozentsatz der Steuer
  belaz_konto         VARCHAR(25),                                                                          --  Kontierung
  belaz_visible       BOOLEAN NOT NULL DEFAULT TRUE,                                                        --  Auf Dokument sichtbar oder nicht?
  belaz_zutxt         TEXT,                                                                                 --  Zusätzlicher Hinweistext
  belaz_zutxt_rtf     TEXT,                                                                                 --  Zusätzlicher Hinweistext (RTF)
  belaz_zutxt_int     TEXT,                                                                                 --  Interner zusatztext (erscheint nicht auf Dokumenten)
  belaz_source_table  VARCHAR(40),                                                                          --  Aus welcher Quelle wurde Abzu hierhin kopiert
  belaz_source_dbrid  VARCHAR(32),                                                                          --  Aus welchem Datensatz wurde Abzu hierhin kopiert
  --                                                                                                      -- Zusatzfelder
  belaz_belegtyp      VARCHAR(3) NOT NULL DEFAULT 'ERG' REFERENCES belegtyp,
  belaz_canrabatt     BOOLEAN NOT NULL DEFAULT FALSE,
  belaz_vorgaenger    INTEGER REFERENCES belegabzu  ON UPDATE CASCADE,
  belaz_ks_abt        VARCHAR(9),
  -- System (tables__generate_missing_fields)
  insert_date         DATE,
  insert_by           VARCHAR(32),
  modified_by         VARCHAR(32),
  modified_date       TIMESTAMP(0)
 );


 -- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
 CREATE TRIGGER belegabzu_set_modified
  BEFORE INSERT OR UPDATE
  ON belegabzu
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();



 CREATE OR REPLACE FUNCTION belegabzu__a_iud() RETURNS TRIGGER AS $$
  DECLARE belegtypRec RECORD;
  BEGIN

   IF BelAbzuTriggerOff() THEN -- Wir machen mal lieber nichts.
      IF Current_User='root' THEN  RAISE NOTICE 'belegabzu__a_iud.Triggers=OFF. belegabzu__a_iud belaz_id: % belaz_dokument_id: % ',
        IfThen(tg_op = 'DELETE', old.belaz_id, new.belaz_id),
        IfThen(tg_op = 'DELETE', old.belaz_dokument_id, new.belaz_dokument_id);  END IF;
      RETURN new;
   END IF;

   IF Current_User='root' THEN RAISE NOTICE 'belegabzu__a_iud belaz_id: % belaz_dokument_id: % ',
        IfThen(tg_op = 'DELETE', old.belaz_id, new.belaz_id),
        IfThen(tg_op = 'DELETE', old.belaz_dokument_id, new.belaz_dokument_id);
   END IF;

   --Meldung ausgeben,wennn Steuersatz ungültig
   IF NOT (tg_op = 'DELETE') THEN
        IF ( (new.belaz_belegtyp = 'ERG') AND (new.belaz_steucode = (SELECT steu_z FROM steutxt WHERE steu_z=new.belaz_steucode AND steu_valid_to<current_date)) ) THEN
                PERFORM PRODAT_TEXT(15810);
        END IF;
   END IF;

   -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen.
   IF (tg_op = 'DELETE') THEN
     SELECT * INTO belegtypRec FROM belegtyp JOIN belegdokument ON belt_typ = beld_belegtyp WHERE old.belaz_dokument_id = beld_id;
   ELSE
     SELECT * INTO belegtypRec FROM belegtyp JOIN belegdokument ON belt_typ = beld_belegtyp WHERE new.belaz_dokument_id = beld_id;
   END IF;


   --Preis nur aktualisieren wenn Belegtyp das erfordert.
   IF belegtypRec.belt_preiscalc THEN

    IF (tg_op = 'DELETE') THEN
      PERFORM TBeleg.Update_DokumentSumme(old.belaz_dokument_id);
      RETURN old;
    ELSE

      IF (tg_op = 'INSERT') THEN
        PERFORM TBeleg.Update_DokumentSumme(new.belaz_dokument_id);
      ELSE
        IF (
                        (new.belaz_betrag <> old.belaz_betrag)
                OR      (new.belaz_anzahl <> old.belaz_anzahl)
                OR      (new.belaz_steuproz <> old.belaz_steuproz)
                OR      (new.belaz_canskonto <> old.belaz_canskonto)
                OR      (new.belaz_canrabatt <> old.belaz_canrabatt)
                ) THEN
          PERFORM TBeleg.Update_DokumentSumme(new.belaz_dokument_id);
        END IF;
      END IF; --End IF Insert

      RETURN new;
    END IF; -- End If Delete

   ELSE --Keine Preisberechnung, irgendwas müssen wir trotzdem returnen.
    IF (tg_op = 'DELETE') THEN
      RETURN old;
    ELSE
      RETURN new;
    END IF;
   END IF; -- End If preiscalc

  END $$ LANGUAGE plpgsql;


  CREATE TRIGGER belegabzu__a_iud
   AFTER INSERT OR UPDATE OR DELETE
   ON belegabzu
   FOR EACH ROW
   EXECUTE PROCEDURE belegabzu__a_iud();
  ---
--

-- Belegpositionen, Adressen, Mengen, Preise .... usw.
CREATE TABLE belegpos(
  -- Grundfelder
  belp_id                       SERIAL NOT NULL PRIMARY KEY,            -- Pos-ID
  belp_belegtyp                 VARCHAR(3) NOT NULL REFERENCES belegtyp,-- Belegtyp der Position
  belp_dokument_id              INTEGER NOT NULL REFERENCES belegdokument_keys ON UPDATE CASCADE ON DELETE CASCADE, -- Dokument auf dem die Position hängt

  --
  belp_krzbesteller             VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --bestelleradresse
  belp_krzrechnung              VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --rechungsadresse
  belp_krzlieferung             VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --lieferadresse
  belp_l_iso                    VARCHAR(5)  REFERENCES laender ON UPDATE CASCADE, -- Ursprungsland
  belp_projektnummer            VARCHAR(50) REFERENCES anl ON UPDATE CASCADE ON DELETE NO ACTION,             -- Projektbezug
  belp_txt                      TEXT,                                   -- Positionszusatztext
  belp_txt_rtf                  TEXT,
  belp_dokutxt                  TEXT,                                   -- Positionszusatztext für das Dokument
  belp_dokutxt_rtf              TEXT,
  belp_referenz                 VARCHAR(50),                            -- Freie Referenznummer auf irgendwas : Lieferscheinnummern, Wareneingangsnummer, Bestellung
                                                                        -- ERG => Lieferschein-Nr. des WE oder Bestellnummer+Pos (wenn Übernahme ohne WE)
                                                                        -- LFS => Bestell-Nr. Kunde aus LA (l_bda) oder Bestell-Nr.Kunde+Pos aus Auftrag (ag_bda und ag_bdapos)
  -- Verknüpfungen
  belp_vorgaenger               INTEGER REFERENCES belegpos ON UPDATE CASCADE ON DELETE SET NULL,               -- Position stammt aus dieser Vorgängerposition, falls aus Beleg übernommen. Sonst Vorgänger = w_wen, ld_id, ag_id etc ....
  belp_ld_id                    INTEGER REFERENCES ldsdok,              -- Bezug zur Vorgänger-Position im Einkauf
  belp_ag_id                    INTEGER REFERENCES auftg,               -- Bezug zur Vorgänger-Position im Verkauf
  belp_w_wen                    INTEGER REFERENCES wendat,              -- Bezug zur Vorgänger-Position im Wareneingang
  belp_l_nr                     INTEGER REFERENCES lifsch,              -- Bezug zur Vorgänger-Position im Warenausgang
  belp_ab_ix                    INTEGER REFERENCES abk,                 -- Bezug zu einer ABK
  belp_a2_id                    INTEGER REFERENCES ab2,                 -- Bezug zu einem Arbeitsgang (ist neu)
  belp_q_nr                     INTEGER REFERENCES qab,                 -- Bezug zu QAB (z.Bsp. bei Retourelieferschein)

  -- Warenbezug
  belp_aknr                     VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE,          --Artikelnummer
  belp_akbez                    VARCHAR(100),                           -- Überschreibbare Artikelbezeichnung
  belp_referenzaknr             VARCHAR(70),
  belp_menge                    NUMERIC(14,6) NOT NULL DEFAULT 0,       -- Menge
  belp_menge_gme                NUMERIC(14,6) NOT NULL DEFAULT 0,       -- Menge in Grundmengeneinheit
  belp_menge_done               NUMERIC(14,6) NOT NULL DEFAULT 0,       -- Menge die geliefert / in Nachfolger uebernommen wurde
  belp_menge_done_gme           NUMERIC(14,6) NOT NULL DEFAULT 0,       -- Menge die geliefert / in Nachfolger uebernommen wurde in Grundmengeneinheit
  belp_mce                      INTEGER NOT NULL CONSTRAINT xtt4064 REFERENCES artmgc, -- Mengeneinheit
  belp_los                      NUMERIC(12,4),                          -- Losgroesse
  belp_gewicht                  NUMERIC(12,4),                          -- Gewicht der Position
  belp_bestandsaktiv            BOOL NOT NULL DEFAULT FALSE,            -- Position kann Lagerbestand verändern
  belp_versandart               INTEGER REFERENCES versart,             -- Versandart, falls irgendwas versendet wurde.
  belp_versandartbem            VARCHAR(100),

  -- Statusfelder
  belp_termin                   DATE,                                   -- Im Lieferschein: Auslieferungs/Lagerabgangsdatum, durch Warenausgangsscan oder Drucken gesetzt
  belp_termin_bestaetigt        DATE,
  belp_termin_verschoben        DATE,
  belp_erstelldatum             DATE,                                   -- Wann wurde Bel.Pos. erstellt
  belp_erledigt                 BOOLEAN NOT NULL DEFAULT FALSE,         -- Position übernommen/abgeschlossen/verbucht?
  belp_storniert                BOOLEAN NOT NULL DEFAULT FALSE,         -- Storniert

  -- System (tables__generate_missing_fields)
  dbrid                         VARCHAR(32) NOT NULL DEFAULT nextval('db_id_seq'),
  insert_date                   DATE,
  insert_by                     VARCHAR(32),
  modified_by                   VARCHAR(32),
  modified_date                 TIMESTAMP(0),

  -- zur internen verwendung
  belp_beleg_id                 INTEGER NOT NULL REFERENCES beleg ON UPDATE CASCADE ON DELETE CASCADE, -- Position ist auf dem Beleg
  belp_pos                      INTEGER NOT NULL,                       -- Positionsnummer
  belp_hauptpos                 INTEGER REFERENCES belegpos,            -- Oberposition falls das Stücklistenzeugs ist

  -- Zahlungsinformationen
  belp_skontofaehig             BOOL NOT NULL DEFAULT TRUE,             -- Für Position kann Skonto berechnet werden
  belp_rabattfaehig             BOOL NOT NULL DEFAULT TRUE,             -- Für Position kann Rabatt berechnet werden
  belp_abzufaehig               BOOL NOT NULL DEFAULT TRUE,             -- Position kann Ab- Zuschläge enthalten
  belp_konto                    VARCHAR(25),                            -- Verrechnungskonto
  belp_kostenstelle             VARCHAR(9) REFERENCES ksv ON UPDATE CASCADE,  -- Kostenstelle auf die der Beleg entfällt
  belp_kurs                     NUMERIC(12,6) NOT NULL DEFAULT 1,       -- Umrechnungskurs zur Stammwährung
  belp_waehr                    VARCHAR(3) NOT NULL CONSTRAINT xtt4056 REFERENCES bewa, -- Währung
  belp_steucode                 INTEGER REFERENCES steutxt ON UPDATE CASCADE, -- Steuersatz für Position

  -- Preisberechnung
  belp_steuproz                 NUMERIC(5,2) NOT NULL DEFAULT 0,        -- Steuersatz in Prozent für Position
  belp_preis                    NUMERIC(16,6) NOT NULL DEFAULT 0,       -- Originalpreis lt. Rechnung (Pro Preiseinheit)
  belp_preis_basis_w            NUMERIC(16,6) NOT NULL DEFAULT 0,       -- Preis in Stammwährung (Pro Preiseinheit)
  belp_preis_gme                NUMERIC(16,6) NOT NULL DEFAULT 0,       -- Originalpreis / GME
  belp_preis_gme_basis_w        NUMERIC(16,6) NOT NULL DEFAULT 0,       -- Preis in Stammwährung / GME
  belp_abzupreis_gme            NUMERIC(16,6) NOT NULL DEFAULT 0,       -- Preis inkl. (Pos. und Beleg)-Zuschläge, Rabatte etc. in Stammwährung / GME
  belp_preiseinheit             NUMERIC(14,4) NOT NULL DEFAULT 1 CONSTRAINT belegpos__chk__preiseinheit CHECK ( belp_preiseinheit > 0),       -- Preiseinheit. Preis gilt pro 100 Stück => Preiseinheit = 100
  belp_netto                    NUMERIC(14,4) NOT NULL DEFAULT 0,       -- Netto-Positionspreis (mit abzu)
  belp_brutto                   NUMERIC(14,4) NOT NULL DEFAULT 0,       -- Brutto-Positionspreis
  belp_netto_basis_w            NUMERIC(14,4) NOT NULL DEFAULT 0,       -- Netto-Positionspreis (mit abzu)
  belp_brutto_basis_w           NUMERIC(14,4) NOT NULL DEFAULT 0,       -- Brutto-Positionspreis
  belp_rabatt                   NUMERIC(5,2) NOT NULL DEFAULT 0,        -- Rabatt in Prozent auf Position
  belp_sumabzu                  NUMERIC(14,2) NOT NULL DEFAULT 0,       -- Gesamtsumme (Brutto)der Positionsabzuschläge zur Anzeige in Oberfläche
  belp_sumBelegAbzu             NUMERIC NOT NULL DEFAULT 0,             -- Anteil der Position an Belegabzuschlaegen (Anteiliger Beleg-NettoUmsatz) in Belegwaehrung.
  belp_preis_mce                INTEGER REFERENCES artmgc,              -- ME auf die sich der Preis bezieht.

  -- Status #7895
  belp_bstat                    VARCHAR(2), -- REFERENCES auftgbs,      -- Statusflags für Auftrag -- wird in X_TableConstraints erstellt
  belp_bstat1                   VARCHAR(2), -- REFERENCES auftgbs,      -- Statusflags für Auftrag -- wird in X_TableConstraints erstellt
  belp_bstat2                   VARCHAR(2), -- REFERENCES auftgbs,      -- Statusflags für Auftrag -- wird in X_TableConstraints erstellt

  -- https://redmine.prodat-sql.de/issues/16057
  -- Verweis auf Verpackungsmaterial der Lieferscheinposition
  belp_lp_id                    integer REFERENCES lifschpack ON UPDATE CASCADE ON DELETE CASCADE
);

 -- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
 CREATE TRIGGER belegpos_set_modified
  BEFORE INSERT OR UPDATE
  ON belegpos
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();

 -- Indizes
    --Belegposition existiert bereits.
    CREATE UNIQUE INDEX xtt10850 ON belegpos (belp_belegtyp, belp_beleg_id, belp_pos);

    CREATE INDEX belegpos_dokid_pos ON belegpos (belp_dokument_id, belp_pos);
    CREATE INDEX belegpos_belid_pos ON belegpos (belp_beleg_id, belp_pos);

    CREATE INDEX belegpos_belp_aknr ON belegpos (belp_aknr);
    CREATE INDEX belegpos_belp_aknr_like ON belegpos (belp_aknr varchar_pattern_ops);

    CREATE INDEX belegpos_belp_krzbesteller ON belegpos (belp_krzbesteller);
    CREATE INDEX belegpos_belp_krzbesteller_like ON belegpos (belp_krzbesteller varchar_pattern_ops);
    CREATE INDEX belegpos_belp_krzrechnung ON belegpos (belp_krzrechnung);
    CREATE INDEX belegpos_belp_krzrechnung_like ON belegpos (belp_krzrechnung varchar_pattern_ops);
    CREATE INDEX belegpos_belp_krzlieferung ON belegpos (belp_krzlieferung);
    CREATE INDEX belegpos_belp_krzlieferung_like ON belegpos (belp_krzlieferung varchar_pattern_ops);

    CREATE INDEX belegpos_erstelldatum ON belegpos (belp_erstelldatum);
    CREATE INDEX belegpos_termin ON belegpos (belp_termin) WHERE belp_termin IS NOT NULL;


    CREATE INDEX belegpos_belp_ag_id ON belegpos(belp_ag_id);

    CREATE INDEX belegpos_belp_ld_id ON belegpos(belp_ld_id);
    CREATE INDEX belegpos_belp_w_wen ON belegpos(belp_w_wen);
    CREATE INDEX belegpos_belp_l_nr  ON belegpos(belp_l_nr);
    CREATE INDEX belegpos_belp_a2_id ON belegpos(belp_a2_id);
    CREATE INDEX belegpos_belp_ab_ix ON belegpos(belp_ab_ix);

    CREATE INDEX belegpos_belp_q_nr ON belegpos(belp_q_nr) WHERE belp_q_nr IS NOT NULL;
    CREATE INDEX belegpos_belp_lp_id ON belegpos(belp_lp_id) WHERE belp_lp_id IS NOT NULL;
 --

 CREATE OR REPLACE FUNCTION belegpos__b_01_iu__aknr() RETURNS TRIGGER AS $$
    BEGIN

        -- INSERT
        IF new.belp_mce IS NULL THEN --Standard ME für Artikel
          new.belp_mce := tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(new.belp_aknr);
        END IF;
        --
        IF tg_op = 'UPDATE' THEN
          -- die meid hat sich nicht geändert, obwohl sich der Artikel geändert hat!
          IF (new.belp_aknr IS DISTINCT FROM old.belp_aknr) AND (new.belp_mce IS NOT DISTINCT FROM old.belp_mce)
          THEN /*wir müssen natürlich auch den Mengencode mit nachziehen!*/
            new.belp_mce := tartikel.me__convertme_for_art__by__mid(new.belp_aknr, old.belp_mce, true);
          END IF;
          IF (new.belp_aknr IS DISTINCT FROM old.belp_aknr) AND (new.belp_preis_mce IS NOT null) AND (new.belp_preis_mce IS NOT DISTINCT FROM old.belp_preis_mce)
          THEN /*wir müssen natürlich auch den Mengencode mit nachziehen!*/
            new.belp_preis_mce := tartikel.me__convertme_for_art__by__mid(new.belp_aknr, old.belp_preis_mce, true);
          END IF;
        END IF;

        -- Der Datensatz wurde mit einer für den Artikel ungültigen Mengeneinheit gespeichert. Tabelle: %, ID: %, Feld: %, ART-Nr.: %, Artmgc-ID: %
        IF NOT TArtikel.me__art__artmgc__m_ids__valid(new.belp_aknr, new.belp_mce)       THEN
           RAISE EXCEPTION '%', Format(lang_text(13795), 'belegpos', new.belp_id::VARCHAR, 'belp_mce'      , new.belp_aknr, new.belp_mce      ::VARCHAR);
        END IF;
        IF NOT TArtikel.me__art__artmgc__m_ids__valid(new.belp_aknr, new.belp_preis_mce) THEN
           RAISE EXCEPTION '%', Format(lang_text(13795), 'belegpos', new.belp_id::VARCHAR, 'belp_preis_mce', new.belp_aknr, new.belp_preis_mce::VARCHAR);
        END IF;
        --
        RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belegpos__b_01_iu__aknr
      BEFORE INSERT OR UPDATE
      OF belp_aknr, belp_mce, belp_preis_mce
      ON belegpos
      FOR EACH ROW
      WHEN (new.belp_aknr IS NOT null) -- wenn null, kommt NOT NULL CONSTRAINT
      EXECUTE PROCEDURE belegpos__b_01_iu__aknr();

 --
  CREATE OR REPLACE FUNCTION belegpos__b_50_i() RETURNS TRIGGER AS $$
    DECLARE belegtypRec RECORD;
            belegRec    RECORD;
    BEGIN

      IF BelPosTriggerOff() THEN -- Wir machen mal lieber nichts.
        IF Current_User='root' THEN
          RAISE NOTICE 'belegpos__b50_i.Triggers=OFF.belegpos__b50_i belp_id: % belp_dokument_id: % ',new.belp_id, new.belp_dokument_id; END IF;
        RETURN new;
      END IF;

      IF Current_User='root' THEN RAISE NOTICE 'belegpos__b50_i belp_id: % belp_dokument_id: % ',new.belp_id, new.belp_dokument_id; END IF;

      -- Die Dokument-Nummer zu unserem Beleg brauchen wir ggf. auch
      SELECT * INTO belegRec FROM belegdokument WHERE beld_id = new.belp_dokument_id;

      IF new.belp_belegtyp IS null THEN new.belp_belegtyp := belegRec.beld_belegtyp; END IF;

      -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen. Also holen wir die Belegartenbeschreibung.
      SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = new.belp_belegtyp;

      -- Belegtyp lässt nur eine 1 zu 1 Abbildung zu Dokumenten zu. Bei Dokument-Insert wurde automatisch ein Beleg angelegt. Den suchen wir jetzt.
      IF (NOT belegtypRec.belt_multipledok) AND (new.belp_beleg_id IS NULL) THEN
        SELECT bel_id INTO new.belp_beleg_id FROM beleg WHERE bel_nummer = belegRec.beld_dokunr AND bel_belegtyp = new.belp_belegtyp;
      END IF;

      IF new.belp_pos IS NULL THEN
        new.belp_pos := next_belegpos(new.belp_belegtyp, belegRec.beld_dokunr);
      END IF;

      new.belp_krzbesteller := COALESCE(new.belp_krzbesteller, belegRec.beld_krzbesteller);
      new.belp_krzlieferung := COALESCE(new.belp_krzlieferung, belegRec.beld_krzlieferung);
      new.belp_krzrechnung  := COALESCE(new.belp_krzrechnung,  belegRec.beld_krzrechnung);
      new.belp_erstelldatum := COALESCE(new.belp_erstelldatum, current_date);
      new.belp_waehr        := COALESCE(new.belp_waehr,        belegRec.beld_waehr);

      IF ( new.belp_a2_id IS NOT NULL ) AND ( new.belp_ab_ix IS NULL) THEN
        SELECT a2_ab_ix INTO new.belp_ab_ix FROM ab2 WHERE a2_id = new.belp_a2_id;
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belegpos__b_50_i
      BEFORE INSERT
      ON belegpos
      FOR EACH ROW
      EXECUTE PROCEDURE belegpos__b_50_i();
 --

 -- Schlüssel für Standardsuche Eingangsrechnungen (Pos), #7157
 CREATE OR REPLACE FUNCTION belegpos__a_iu_erg__keywordsearch() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM TSystem.kws_create_keywords(new.*);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__a_iu_erg__keywordsearch
    AFTER INSERT OR UPDATE
    OF belp_referenz, belp_akbez, belp_referenzaknr, belp_projektnummer
    ON belegpos
    FOR EACH ROW
    WHEN (new.belp_belegtyp = 'ERG') -- Nur bei Eingangsrechnungsposition reagieren.
    EXECUTE PROCEDURE belegpos__a_iu_erg__keywordsearch();
 --

 -- Hauptsaechlich Preise und Mengeneinheiten umrechnen
 CREATE OR REPLACE FUNCTION belegpos__b_50_iu() RETURNS TRIGGER AS $$
  DECLARE uf_me          NUMERIC;
    uf_preis    NUMERIC;
        rec         RECORD;
        belegtypRec     RECORD;
        mybelp_menge_uf1  NUMERIC;    -- #7326
        mybelp_vp_uf1    NUMERIC;    -- #7326
  BEGIN

      IF BelPosTriggerOff() THEN -- Wir machen mal lieber nichts.
            IF Current_User='root' THEN  RAISE NOTICE 'belegpos__b50_iu.Triggers=OFF. belegpos__b50_iu belp_id: % belp_dokument_id: %', new.belp_id, new.belp_dokument_id;  END IF;
            RETURN new;
      END IF;

      IF Current_User='root' THEN RAISE NOTICE 'belegpos__b50_iu belp_id: % belp_dokument_id: %', new.belp_id, new.belp_dokument_id;  END IF;

      -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen.
      SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = new.belp_belegtyp;

      SELECT COALESCE(m_uf,1) INTO uf_me FROM artmgc    WHERE m_id = new.belp_mce; -- UF der Menge
      SELECT COALESCE(m_uf,1) INTO uf_preis FROM artmgc WHERE m_id = COALESCE(new.belp_preis_mce, new.belp_mce); -- #7326 -- UF des Preises. Wenn keine Preis-ME dann Positions-ME verwenden

      IF COALESCE(uf_preis,0)=0 THEN
            uf_preis:=1;
      END IF;

     --Kursänderung? Belegpositions- und Abzupreise neu berechnen, falls Belegtyp Preisberechnung macht
      IF ((TG_OP = 'UPDATE') AND belegtypRec.belt_preiscalc) THEN
        IF (old.belp_kurs <> new.belp_kurs) AND (new.belp_waehr <> old.belp_waehr) THEN
          new.belp_preis := new.belp_preis * (old.belp_kurs/new.belp_kurs);

          --Betrag neu ausrechnen und unterbinden, dass Belposabzu sofort hierhin zurückschreibt
          UPDATE belegposabzu SET belpaz_betrag = belpaz_betrag * (old.belp_kurs/new.belp_kurs) WHERE belpaz_belegpos_id = new.belp_id;
         END IF;
      END IF;

      new.belp_projektnummer := UPPER (new.belp_projektnummer);
      new.belp_kurs := COALESCE(new.belp_kurs,1);
      new.belp_menge := COALESCE(new.belp_menge,0);
      new.belp_preis := COALESCE(new.belp_preis,0);
      new.belp_preiseinheit := COALESCE(new.belp_preiseinheit,0);
      new.belp_netto:=COALESCE(new.belp_netto,0);
      new.belp_brutto:=COALESCE(new.belp_brutto,0);

      -- Belegmenge von Beleg-ME => Grundmengeneinheit
      new.belp_menge_gme  := tartikel.me__menge__in__menge_uf1(new.belp_mce, new.belp_menge);

      -- Menge erledigt von Grundmengeneinheit => Beleg-ME
      new.belp_menge_done := COALESCE( new.belp_menge_done_gme *  uf_me , 0);

      -- Alle Preise, ausser belp_preis selbst, beruecksichtigen die Preiseinheit. (macht keinen Mehraufwand => Auch aktiv wenn keine Preisberechnung)
      new.belp_preis_gme := COALESCE( new.belp_preis * uf_preis / IfThen(new.belp_preiseinheit=0, 1, new.belp_preiseinheit), 0); -- Preis * Menge / Preiseinheit
      new.belp_preis_basis_w := new.belp_preis * new.belp_kurs / IfThen(new.belp_preiseinheit=0, 1, new.belp_preiseinheit);
      new.belp_preis_gme_basis_w := new.belp_preis_gme * new.belp_kurs;



      --Ab-/Zuschläge = (Anzahl*Betrag * [Steuerprozent])

      SELECT SUM(COALESCE(belpaz_betrag, 0) * belpaz_anzahl) AS abzunetto,
             SUM(COALESCE(belpaz_betrag, 0) * belpaz_anzahl * (1+COALESCE(belpaz_steuproz,0)/100)) AS abzubrutto INTO rec
        FROM belegposabzu
        WHERE belpaz_belegpos_id = new.belp_id;

      new.belp_sumabzu:=COALESCE(rec.abzubrutto,0);

      --PosPreis inklusive Ab- und Zuschlägen und Rabatten für die Position (ggf. in Fremdwährung) (macht keinen Mehraufwand => Auch aktiv wenn keine Preisberechnung)
      new.belp_netto  := (new.belp_preis_gme_basis_w * new.belp_menge_gme
                            * IfThen( (new.belp_rabattfaehig),(1 - new.belp_rabatt / 100), 1)
                            / IfThen(new.belp_kurs=0,1,new.belp_kurs)
                            )+ COALESCE(rec.abzunetto,0);
      new.belp_brutto := (new.belp_preis_gme_basis_w * new.belp_menge_gme
                            * (1 + new.belp_steuproz / 100)
                            * IfThen( (new.belp_rabattfaehig),(1 - new.belp_rabatt / 100), 1)
                            / IfThen(new.belp_kurs=0,1,new.belp_kurs)
                            )+ COALESCE(rec.abzubrutto,0);

      new.belp_netto_basis_w := new.belp_netto * new.belp_kurs;
      new.belp_brutto_basis_w := new.belp_brutto * new.belp_kurs;

      new.belp_abzupreis_gme :=  --Preis mit PosRabatt pro ME + AbzuAnteil
            IfThen ( (new.belp_menge_gme = 0), 0, new.belp_netto_basis_w / do1If0(new.belp_menge_gme))
            + ( COALESCE( new.belp_sumBelegAbzu,0) * new.belp_kurs ) / do1If0(new.belp_menge_gme);



      IF new.belp_belegtyp = 'LFS' THEN
        --Erledigt Status setzen, wenn übernommene Menge >= Menge und wir überhaupt ne Menge haben.
        --Aber nur bei Änderung der fakturierten Mengen, damit man das zur Not auch händisch setzen kann.
        --Zumachen geht automatisch, aufmachen nur haendisch oder per Rueckfrage bei Aenderung Belzeil in Faktura.
        IF ( TG_OP = 'INSERT') OR (( TG_OP = 'UPDATE') AND (old.belp_menge_done_gme <> new.belp_menge_done_gme)) THEN
            new.belp_erledigt := IfThen ( ( Round( COALESCE( new.belp_menge_done_gme,0) ,2) >= ROUND( COALESCE( new.belp_menge_gme,0),2) ) AND (NOT new.belp_menge_gme = 0)
                            , TRUE, FALSE);

        END IF;
      END IF;

      RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__b_50_iu
    BEFORE INSERT OR UPDATE
    ON belegpos
    FOR EACH ROW
    EXECUTE PROCEDURE belegpos__b_50_iu();
 --

--
CREATE OR REPLACE FUNCTION belegpos__b_65_i__lfs__belp_gewicht() RETURNS TRIGGER AS $$
  DECLARE
      _artikel_gewicht_uf1  numeric;
  BEGIN
    -- Bei Anlage einer neuen Lieferschein-Pos. wird automatisch das Artikelgewicht ermittelt.
    -- vgl. FUNCTION TBeleg.LiefPosFromLagAb


    -- Wenn noch nicht bei Erstellung angg.
    IF new.belp_gewicht IS NULL THEN

        -- Gewicht pro GME laut Artikelstamm.
        _artikel_gewicht_uf1 := ak_gewicht FROM art WHERE ak_nr = new.belp_aknr;

        -- Wert nur, wenn Gewicht im Artikel angg. (kann NULL werden).
        new.belp_gewicht := new.belp_menge_gme * _artikel_gewicht_uf1;

    END IF;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__b_65_i__lfs__belp_gewicht
    BEFORE INSERT
    ON belegpos
    FOR EACH ROW
    WHEN ( new.belp_belegtyp = 'LFS' )
    EXECUTE PROCEDURE belegpos__b_65_i__lfs__belp_gewicht();
--

 -- Anforderung ursprungserklärung Referenzartikel-Nummer holen
  CREATE OR REPLACE FUNCTION belegpos__b_70_iu__custombeleg() RETURNS TRIGGER AS $$
  DECLARE referenzaknr VARCHAR;
  BEGIN
    --nur für LUE: Ziel vorhandene Referenzartikel-Nummer speichern für Belegausgabe
    referenzaknr := (SELECT e_best FROM epreis LEFT JOIN belegdokument ON beld_id = new.belp_dokument_id
                    WHERE e_aknr = new.belp_aknr AND e_lkn = beld_krzlieferung ORDER BY e_id DESC LIMIT 1);

    IF (tg_op='INSERT') THEN
      IF (new.belp_aknr IS NOT NULL) AND (referenzaknr IS NOT NULL) THEN
        new.belp_referenzaknr := referenzaknr;
      END IF;
    END IF;

    IF (tg_op='UPDATE') THEN
      IF (new.belp_aknr IS NOT NULL) AND (referenzaknr IS NOT NULL) AND (new.belp_aknr IS DISTINCT FROM old.belp_aknr) THEN
        new.belp_referenzaknr := referenzaknr;
      END IF;
    END IF;
     --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__b_70_iu__custombeleg
   BEFORE INSERT OR UPDATE
   ON belegpos
   FOR EACH ROW
   WHEN (new.belp_belegtyp = 'LUE')
   EXECUTE PROCEDURE belegpos__b_70_iu__custombeleg();
 --

 -- Uebernommene Mengen zurueckschreiben
 CREATE OR REPLACE FUNCTION belegpos__a_50_iud() RETURNS TRIGGER AS $$
    DECLARE belegtypRec RECORD;
    BEGIN

      IF BelPosTriggerOff() THEN -- Wir machen mal lieber nichts.

        /* --geht so nicht, ifthen inhalt wird vorher geprüft, old bzw. new existiert nicht

        IF Current_User='root' THEN  RAISE NOTICE 'belegpos__a50_iud.Triggers=OFF.belegpos__a50_iud belp_id: % belp_dokument_id: % ',
            IfThen(tg_op = 'DELETE', old.belp_id, new.belp_id),
            IfThen(tg_op = 'DELETE', old.belp_dokument_id, new.belp_dokument_id); END IF;
        */
        RETURN new;

      END IF;

      /* --geht so nicht, ifthen inhalt wird vorher geprüft, old bzw. new existiert nicht

      IF Current_User='root' THEN RAISE NOTICE 'belegpos__a50_iud belp_id: % belp_dokument_id: % ',
            IfThen(tg_op = 'DELETE', old.belp_id, new.belp_id),
            IfThen(tg_op = 'DELETE', old.belp_dokument_id, new.belp_dokument_id);
      END IF;
      */

      -- Belegart holen.
      IF (tg_op = 'DELETE') THEN
        SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = old.belp_belegtyp;
      ELSE
        SELECT * INTO belegtypRec FROM belegtyp WHERE belt_typ = new.belp_belegtyp;
      END IF;


      IF (tg_op = 'DELETE') THEN
        IF (old.belp_belegtyp = 'ERG') THEN
          PERFORM TBeleg.Update_ldsdokStkf(old.belp_ld_id);
          PERFORM TBeleg.Update_wendatStkf(old.belp_w_wen);
        END IF;

        IF (old.belp_belegtyp = 'LFS') THEN
          PERFORM TBeleg.LFS_Update_Auftg(old.belp_ag_id);
          --Lieferscheinpos. löschen => Lagerabgang wieder für Übernahmen öffnen. l_belp_id ist ON DELETE SET NULL. lifsch trigger setzt dann l_inliefdok zurück.
        END IF;


        IF belegtypRec.belt_preiscalc THEN
          PERFORM TBeleg.Update_DokumentSumme(old.belp_dokument_id);
        END IF;

        RETURN old; --ENDE DELETE

      ELSE

        IF (tg_op = 'INSERT') THEN

          IF (new.belp_belegtyp = 'ERG') THEN
            PERFORM TBeleg.Update_ldsdokStkf(new.belp_ld_id);
            PERFORM TBeleg.Update_wendatStkf(new.belp_w_wen);
          END IF;


          IF (new.belp_belegtyp = 'LFS') THEN


          PERFORM TBeleg.LFS_Update_Auftg(new.belp_ag_id) WHERE NOT TSystem.execution_code__is_disabled('belegpos__a_50_iud.LFS_Update_Auftg');

            --Neue Lieferposition eingefuegt. Wenn die Verrechnungsfaehig und nicht gleich erledigt ist, muessen wir Verbucht Status im Lieferdokument zuruecksetzen
            IF (NOT new.belp_storniert) AND (NOT new.belp_erledigt) THEN
              UPDATE belegdokument SET beld_verbucht = FALSE WHERE new.belp_dokument_id = beld_id AND beld_verbucht;
            END IF;

          END IF;


          IF belegtypRec.belt_preiscalc THEN
                PERFORM TBeleg.Update_DokumentSumme(new.belp_dokument_id);
          END IF;

          -- Ursprungserklärung #8355 #7951
          -- Wenn der Artikel noch nicht überwacht ist, dann soll er spätestens überwacht werden, wenn der Anforderungsbeleg erstellt wurde
          IF (new.belp_belegtyp = 'LUE') AND (SELECT TRUE FROM artinfo WHERE ain_ak_nr = new.belp_aknr AND ain_lue_req IS NULL) THEN
              UPDATE artinfo SET ain_lue_req = TRUE WHERE ain_ak_nr = new.belp_aknr;
          END IF;

          --ENDE INSERT
        ELSE  --UPDATES

          --Preise nur durchrechnen, wenn es sein muss.
          IF belegtypRec.belt_preiscalc THEN
            IF (    (new.belp_menge <> old.belp_menge)
                    OR      (new.belp_preis <> old.belp_preis)
                    OR      (new.belp_preiseinheit <> old.belp_preiseinheit)
                    OR      (new.belp_preis_gme <> old.belp_preis_gme)
                    OR      (new.belp_kurs <> old.belp_kurs)
                    OR      (new.belp_rabatt <> old.belp_rabatt)
                    OR      (new.belp_steuproz <> old.belp_steuproz)
                    OR      (new.belp_rabattfaehig <> old.belp_rabattfaehig)
                    OR      (new.belp_skontofaehig <> old.belp_rabattfaehig)
                    ) THEN
              PERFORM TBeleg.Update_DokumentSumme(new.belp_dokument_id);
            END IF;
          END IF;


          /* MENGENAENDERUNG                        *********************************************/
          IF (new.belp_menge_gme <> old.belp_menge_gme) OR (old.belp_menge_done_gme <> new.belp_menge_done_gme) THEN

            --EINGANGSRECHNUNG
            IF (new.belp_belegtyp = 'ERG') THEN

              PERFORM TBeleg.Update_ldsdokStkf(new.belp_ld_id);
              IF ( old.belp_ld_id <> new.belp_ld_id ) THEN PERFORM TBeleg.Update_ldsdokStkf(old.belp_ld_id); END IF;

        PERFORM TBeleg.Update_wendatStkf(new.belp_w_wen);
              IF (old.belp_w_wen <> new.belp_w_wen) THEN   PERFORM TBeleg.Update_wendatStkf(old.belp_w_wen); END IF;

            END IF;


            --LIEFERSCHEIN
            IF (new.belp_belegtyp = 'LFS') THEN

              --Liefermengen und Fakturierte Mengen anpassen
              PERFORM TBeleg.LFS_Update_Auftg(new.belp_ag_id);
              IF (old.belp_ag_id <> new.belp_ag_id) THEN   PERFORM TBeleg.LFS_Update_Auftg(old.belp_ag_id); END IF;


            END IF;  --ENDE IF 'LFS'

          END IF; -- Ende MengenAenderung

          /* ERLEDIGT STATUS -- wird jetzt im eigenen Trigger ** belegpos__a_80_iu__erledigt_storniert ** behandelt, JM, #7895
          IF (old.belp_erledigt <> new.belp_erledigt) OR (old.belp_storniert <> new.belp_storniert )THEN

            --LIEFERSCHEIN
            IF (new.belp_belegtyp = 'LFS') THEN
              --Lieferposition fertig fakturiert oder Fakt. aufgehoben.eingefuegt. Also Fakturiert-Status im Lieferdokument zuruecksetzen.
              UPDATE belegdokument
                SET beld_verbucht = NOT EXISTS(SELECT true FROM belegpos WHERE belp_dokument_id = new.belp_dokument_id AND NOT belp_erledigt AND NOT belp_storniert)
              WHERE new.belp_dokument_id = beld_id;
            END IF;
          END IF;
          */

        --ENDE UPDATE
        END IF;


        /* INSERT OR UPDATE landet beides hier, Delete nicht.       *********************************************/

        --Verschlagwortung Dokumentverwaltung
        IF (new.belp_belegtyp = 'ERG') THEN
            UPDATE picndoku SET pd_id=pd_id FROM eingrechdokument WHERE new.belp_dokument_id=beld_id AND pd_dbrid=eingrechdokument.dbrid;
        END IF;

        --Meldung ausgeben,wennn Steuersatz ungültig
        IF (new.belp_belegtyp = 'ERG') AND (new.belp_steucode = (SELECT steu_z FROM steutxt WHERE steu_z=new.belp_steucode AND steu_valid_to<current_date)) THEN
            PERFORM PRODAT_TEXT(15810);
        END IF;


        IF (new.belp_belegtyp = 'LFS') THEN
          UPDATE picndoku SET pd_id=pd_id FROM belegdokument WHERE new.belp_dokument_id=beld_id AND pd_dbrid=belegdokument.dbrid;
        END IF;

        RETURN new;

      END IF;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__a_50_iud
    AFTER INSERT OR UPDATE OR  DELETE
    ON belegpos
    FOR EACH ROW
    EXECUTE PROCEDURE belegpos__a_50_iud();
 --

 -- Externen Text an Rechnungspositionen weitergeben, wenn dort noch nicht geändert und nicht definitv. Ansonsten Meldung.
 CREATE OR REPLACE FUNCTION belegpos__a_60_u__belp_dokutxt() RETURNS TRIGGER AS $$
  DECLARE infotxt TEXT;
  BEGIN -- WHEN (old.belp_dokutxt IS DISTINCT FROM new.belp_dokutxt)

    -- In Rechnung soll immer Text aus Artikelstamm genommen werden #8296
    IF TSystem.Settings__GetBool('bzUseArtTxt') THEN
        RETURN new;
    END IF;
    -- Meldung für nicht veränderbare Rechnungspositionen, bei denen der externe Text schon unterschiedlich ist oder das Dokument definitiv.
    IF EXISTS(SELECT true FROM belzeil_grund JOIN belkopf ON be_bnr = bz_be_bnr WHERE bz_belp_id = new.belp_id AND bz_zubez IS DISTINCT FROM new.belp_dokutxt AND (bz_zubez IS DISTINCT FROM old.belp_dokutxt OR be_def)) THEN
        infotxt:=lang_text(16340)||E'\n\n'||lang_text(379)||E':\n'|| -- Der Dokumenttext im Folgedokument wurde nicht angepasst ... Faktura: ...
            (SELECT array_to_string(ARRAY( -- Stringliste aller Rechnungspostionen
                SELECT be_prof||' '||bz_be_bnr||' '||lang_text(164)||' '||bz_pos
                FROM belzeil_grund JOIN belkopf ON be_bnr = bz_be_bnr
                WHERE bz_belp_id = new.belp_id
                  AND bz_zubez IS DISTINCT FROM new.belp_dokutxt -- Ziel soll geändert werden
                  AND (bz_zubez IS DISTINCT FROM old.belp_dokutxt OR be_def) -- war aber vorher abweichend
                ORDER BY be_prof, bz_be_bnr, bz_pos
                ), E'\n'));
    END IF;

    IF infotxt IS NOT NULL THEN
        PERFORM PRODAT_TEXT(infotxt);
    END IF;

    -- Rechnungspositionen, die automatisch synchron gehalten werden.
    UPDATE belzeil_grund SET
      bz_zubez = new.belp_dokutxt,
      bz_zubez_rtf = new.belp_dokutxt_rtf
    FROM belkopf
    WHERE bz_be_bnr = be_bnr
      AND bz_belp_id = new.belp_id
      AND NOT be_def
      AND bz_zubez IS NOT DISTINCT FROM old.belp_dokutxt -- waren die Texte vorher gleich
      AND bz_zubez IS DISTINCT FROM new.belp_dokutxt;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__a_60_u__belp_dokutxt
   AFTER UPDATE
   OF belp_dokutxt
   ON belegpos
   FOR EACH ROW
   WHEN (old.belp_dokutxt IS DISTINCT FROM new.belp_dokutxt)
   EXECUTE PROCEDURE belegpos__a_60_u__belp_dokutxt();
 --

 -- Eintragung DLZ
 CREATE OR REPLACE FUNCTION belegpos__a_40_iu__dlz() RETURNS TRIGGER AS $$
  DECLARE auftgbdat  DATE;
          bdea_first DATE;
  BEGIN
    IF new.belp_termin > COALESCE((SELECT MAX(belp_termin) FROM belegpos WHERE belp_ag_id=new.belp_ag_id AND NOT belp_id=belp_id), '1.1.1970'::DATE) THEN

        bdea_first=(SELECT min(ba_anf)::DATE FROM bdea WHERE ba_ix IN (
            SELECT DISTINCT tplanterm.get_all_child_abk(ld_abk) FROM ldsdok WHERE ld_ag_id=new.belp_ag_id));

        PERFORM -- Gsamtdurchlaufzeit (Bestelldatum bis Lieferschein, Tage) eintragen
           TRecnoParam.Set('auftg.dlz.Summe'    , auftg.dbrid, (new.belp_termin::date-ag_bdat   )::INTEGER),
           -- Fertigungszeit (1. Stempelung bis Lieferschein, Tage) eintragen
           TRecnoParam.Set('auftg.dlz.Fertigung', auftg.dbrid, (new.belp_termin::date-bdea_first)::INTEGER)
          FROM auftg WHERE ag_id = new.belp_ag_id;
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__a_40_iu__dlz
    AFTER INSERT OR UPDATE
    OF belp_termin
    ON belegpos
    FOR EACH ROW
    WHEN (new.belp_belegtyp = 'LFS')
    EXECUTE PROCEDURE belegpos__a_40_iu__dlz();
--

-- ausgelagerte Triggerfunktion für belegpos__a_id__picndoku, erleichtert das Debugging
CREATE OR REPLACE FUNCTION belegpos__a_id__picndoku( _belp belegpos, _isInsert boolean ) RETURNS void AS $$
  DECLARE
    _belegnr varchar;
    _ab_ixs integer[];
    _ab_ix integer;
    _abks integer[];
    _l_nrs integer[];
    _l_nrs_kauf integer[];
    _l_nr integer;
    _lifsch lifsch;
    _picndoku_dbrids varchar[];
    _picndoku_dbrid varchar;
  BEGIN

    -- verschlagwortet Chargenzeugnisse mit den Lieferscheinen der Produkte, in welchen die Chargen verbaut sind
    -- bzw. Kaufaufträgen versendet wurden

    IF _belp.belp_belegtyp <> 'LFS' THEN
      RETURN;
    END IF;

    _belegnr := beld_dokunr FROM belegdokument WHERE beld_id = _belp.belp_dokument_id;

    -- falls wir ein Produktionsauftrag sind, suchen wir die ABK aus dem Lagerort, von dem wir abgebucht wurden.
    -- der Lagerzugang dieses Lagerortes muss einen Bezug zur internen Bestellung haben
    -- über das Keyword dieses ABK-Indexes müßten wir *alle* quality_we's finden mittels get_all_child_abk unterabks suchen
    _ab_ixs := array_agg( ld_abk ) FROM ldsdok WHERE ld_ag_id = _belp.belp_ag_id;

    IF _ab_ixs IS NOT null THEN
      FOREACH _ab_ix IN ARRAY _ab_ixs LOOP
        _abks := _abks || array_agg( DISTINCT get_all_child_abk ) FROM tplanterm.get_all_child_abk( _ab_ix );
      END LOOP;
    END IF;

    -- alle Lagerabgänge aller ABKs der Belegposition
    SELECT array_agg( DISTINCT l_nr ) INTO _l_nrs FROM lifsch WHERE l_ab_ix = ANY( _abks );

    -- Auch eigene Liefernummer hinzufügen (Kaufauftrag)

    SELECT array_agg( l_nr ) INTO _l_nrs_kauf FROM lifsch
      WHERE l_belp_id = _belp.belp_id;

    -- Zur Sicherheit bei Erzeugung des Lieferscheins aus dem Auftrag heraus
    IF array_length( _l_nrs, 1 ) IS null THEN
      SELECT array_agg( DISTINCT l_nr ) INTO _l_nrs FROM lifsch WHERE l_ag_id = _belp.belp_ag_id;
    END IF;

    _l_nrs := _l_nrs || _l_nrs_kauf;

    --suche Zeugnisse zu l_nr
    IF _l_nrs IS NOT null THEN
      FOREACH _l_nr IN ARRAY _l_nrs LOOP
        SELECT * FROM lifsch INTO _lifsch WHERE l_nr = _l_nr;
        _picndoku_dbrids := picndocu_dbrid__lifsch__chargenzeugnisse__get( _lifsch );
        IF NOT _isInsert THEN  -- DELETE

          -- Löschung der Verknüpfungen
          DELETE FROM recnokeyword
          WHERE
                  r_tablename = 'picndoku'
              AND r_kategorie = 'belegdokument'
              -- Es müssen die Fälle "ganzer Lieferschein gelöscht" und "Belegposition gelöscht" behandelt werden.
              AND ( NOT EXISTS( SELECT 1 FROM belegdokument WHERE beld_dokunr = r_descr ) OR r_descr = _belegnr )
              AND r_dbrid = ANY( _picndoku_dbrids );

        ELSE -- INSERT

          -- Einfügen der Verknüpfungen
          IF _picndoku_dbrids IS NOT null THEN
            FOREACH _picndoku_dbrid IN ARRAY _picndoku_dbrids LOOP
              PERFORM CreateRecNoKeyword( 'belegdokument', _belegnr, _picndoku_dbrid );
            END LOOP;
          END IF;

        END IF;
      END LOOP;
    END IF;

END $$ LANGUAGE plpgsql STRICT;
--

/* NICHT LÖSCHEN
-- Es handelt sich hier um eine Funktion, welche bei KB nur manuell ausgeführt werden soll und nicht offizieller
--   Bestandteil des Codes ist. Insbesendere soll sie in kein DBU.

-- verknüpft Chargenzeugnisse mit Lagerabgängen, ABKS und Liefrscheinpositionen nachträglich ab einem bestimmten Zeitpunkt
CREATE OR REPLACE FUNCTION chargenzeugnisse__verknuepfen__batch( _zeitpunkt date ) RETURNS void AS $$
BEGIN

  CREATE OR REPLACE FUNCTION chargenzeugnisse__verknuepfen__lifsch( _lifsch lifsch ) RETURNS void AS $inner$
  DECLARE
    _picndoku_dbrids varchar[];
  _picndoku_dbrid varchar;
  BEGIN
  IF _lifsch.l_abgg > 0 THEN

    -- Hinzufügen der Verknüpfungen + Verschlagwortung an Zeugnisse, die an der ausgebuchten Charge hängen

    _picndoku_dbrids := picndocu_dbrid__lifsch__chargenzeugnisse__get( _lifsch );
    IF _picndoku_dbrids IS NOT null THEN
      FOREACH _picndoku_dbrid IN ARRAY _picndoku_dbrids LOOP

      PERFORM CreateRecNoKeyword( 'lifsch', _lifsch.l_nr, _picndoku_dbrid );

      IF _lifsch.l_ab_ix IS NOT null THEN
        PERFORM CreateRecNoKeyword( 'abk', _lifsch.l_ab_ix, _picndoku_dbrid );
      END IF;

      END LOOP;
    END IF;

  END IF;
  END;
  $inner$ language plpgsql;

  -- ABKs und Lagerabgänge mit Chargenzeugnissen verknüpfen
  PERFORM chargenzeugnisse__verknuepfen__lifsch( lifsch ) FROM lifsch WHERE insert_date >= _zeitpunkt AND l_ab_ix IS NOT null AND l_lgchnr <> '';

  -- Lieferscheinpositionen mit Chargenzeugnissen verknüpfen
  PERFORM belegpos__a_id__picndoku( belegpos, true ) FROM belegpos WHERE insert_date >= _zeitpunkt AND belp_belegtyp = 'LFS';

END $$ LANGUAGE plpgsql VOLATILE STRICT;
--
*/


CREATE OR REPLACE FUNCTION belegpos__a_id__picndoku() RETURNS TRIGGER AS $$
  BEGIN

    -- verschlagwortet Chargenzeugnisse mit den Lieferscheinen der Produkte, in welchen die Chargen verbaut sind
    -- bzw. Kaufaufträgen versendet wurden

    IF tg_op = 'INSERT' THEN
      PERFORM belegpos__a_id__picndoku( new, true );
      RETURN new;
    ELSE -- DELETE
      PERFORM belegpos__a_id__picndoku( old, false );
      RETURN old;
    END IF;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegpos__a_id__picndoku
    AFTER INSERT OR DELETE
    ON belegpos
    FOR EACH ROW
    EXECUTE PROCEDURE belegpos__a_id__picndoku();
--

CREATE OR REPLACE FUNCTION next_belegpos(belegtyp VARCHAR, belegnummer VARCHAR) RETURNS INTEGER AS $$
DECLARE  belegtyp       ALIAS FOR $1;
         belegnummer    ALIAS FOR $2;
         belegid        INTEGER;
         I              INTEGER;
BEGIN
  SELECT bel_id INTO belegid FROM beleg WHERE bel_belegtyp = belegtyp AND bel_nummer = belegnummer;
  I:=max(belp_pos)+1 FROM belegpos WHERE belp_beleg_id=belegid;

  IF I IS NULL THEN
     I:=1;
  END IF;

  RETURN I;
END$$LANGUAGE plpgsql;


--Bei Erstellen oder Aktualisierung einer Auswärtsrechnung, aktualisieren wir den Rueckmelde-Eintrag
CREATE OR REPLACE FUNCTION belegpos__a_40_iu__update_rm() RETURNS TRIGGER AS $$
DECLARE awap VARCHAR; --Auswärtsarbeitspaket
BEGIN

  -- Auswärts-AP des Arbeitsganges holen
  SELECT a2_aknr INTO awap FROM ab2 WHERE a2_id = new.belp_a2_id AND a2_ausw;
  -- Entspricht nicht dem Arbeitspaket aus ABK. Oder hat kein Auswärtspaket? Evtl. Speditionsrechnung oder sowas? Ignorieren wir auf jeden Fall.
  IF (awap IS NULL) OR (new.belp_aknr <> awap) THEN
    RETURN new;
  END IF;

  PERFORM tplanterm.update_rm_ausw(new.belp_a2_id, new.belp_w_wen, new.belp_id);
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER belegpos__a_40_iu__update_rm
  AFTER INSERT OR UPDATE OF belp_a2_id, belp_preis,belp_preis_gme,belp_menge,belp_menge_gme,belp_erstelldatum
  ON belegpos
  FOR EACH ROW  -- Nur wenn Rechnung ohne Wareneingang. Bei Auswaerts-Ruecklieferung macht das der Wendat Trigger.
  WHEN ((new.belp_belegtyp = 'ERG') AND (new.belp_a2_id IS NOT NULL) AND (new.belp_w_wen IS NULL))
EXECUTE PROCEDURE belegpos__a_40_iu__update_rm();


--Gibt die nächste freie Positionsnummer zurück. Bei usedokumentpos wird nächste Position auf dem Dokument genutzt, sonst nächste freie Position im zugehörigen Beleg
CREATE OR REPLACE FUNCTION next_belegpos(belegtyp VARCHAR,satzid INTEGER, usedokumentpos BOOLEAN) RETURNS INTEGER AS $$
DECLARE  I INTEGER;
BEGIN

  IF usedokumentpos THEN
     I:=max(belp_pos)+1 FROM belegpos WHERE belp_dokument_id=satzid;
  ELSE
     I:=max(belp_pos)+1 FROM belegpos WHERE belp_beleg_id=satzid;
  END IF;

  IF I IS NULL THEN
     I:=1;
  END IF;

  RETURN I;
END$$LANGUAGE plpgsql;





-- Ab- / Zuschläge Positionsbezogen
CREATE TABLE belegposabzu (                                                                      -- VIEWFELDER
  belpaz_id            SERIAL NOT NULL PRIMARY KEY,                                                --  ID des Zuschlags
  belpaz_type          VARCHAR(1)  DEFAULT 'E',                                                    --  Zuschlagstyp, E-Einmalig, P-Position, M-Per ME
  belpaz_pos           INTEGER,                                                                    --  Position
  belpaz_abzu_id       INTEGER NOT NULL REFERENCES abzu  ON UPDATE CASCADE,                        --  ID der Vorgabe
  belpaz_belegpos_id   INTEGER NOT NULL REFERENCES belegpos ON UPDATE CASCADE ON DELETE CASCADE,   --  ID des Datensatzes an dem der Zuschlag hängt
  belpaz_anzahl        NUMERIC(12,4) NOT NULL DEFAULT 1,                                           --  Anzahl / Menge
  belpaz_betrag        NUMERIC(12,4) NOT NULL DEFAULT 0,                                           --  Betrag in Währung des Parent-Datensatzes
  belpaz_prozent       NUMERIC(12,4) DEFAULT 0,                                                    --  Prozentualer Zuschlag auf Basis des Parent-Betrags
  belpaz_canskonto     BOOLEAN NOT NULL DEFAULT TRUE,                                              --  Gilt Skonto für den Zuschlag?
  belpaz_steucode      INTEGER REFERENCES steutxt ON UPDATE CASCADE,                               --  Steuercode (meist gleich Parent)
  belpaz_steuproz      NUMERIC(5,2) NOT NULL DEFAULT 0,                                            --  Prozentsatz der Steuer
  belpaz_konto         VARCHAR(25),                                                                --  Kontierung
  belpaz_sichtbar      BOOLEAN NOT NULL DEFAULT TRUE,                                              --  Auf Dokument sichtbar oder nicht?
  belpaz_zutxt         TEXT,                                                                       --  Zusätzlicher Hinweistext
  belpaz_zutxt_rtf     TEXT,                                                                       --  Zusätzlicher Hinweistext (RTF)
  belpaz_zutxt_int     TEXT,                                                                       --  Interner zusatztext (erscheint nicht auf Dokumenten)
  belpaz_source_table  VARCHAR(40),                                                                --  Aus welcher Quelle wurde Abzu hierhin kopiert
  belpaz_source_dbrid  VARCHAR(32),                                                                --  Aus welchem Datensatz wurde Abzu hierhin kopiert
  --                                                                                             -- ZUSATZFELDER
  belpaz_belegtyp      VARCHAR(3) NOT NULL DEFAULT 'ERG' REFERENCES belegtyp,
  belpaz_canrabatt     BOOLEAN NOT NULL DEFAULT FALSE,
  belpaz_vorgaenger    INTEGER REFERENCES belegposabzu  ON UPDATE CASCADE,
  -- System (tables__generate_missing_fields)
  insert_date          DATE,
  insert_by            VARCHAR(32),
  modified_by          VARCHAR(32),
  modified_date        TIMESTAMP(0)
 );
--

-- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
CREATE TRIGGER belegposabzu_set_modified
  BEFORE INSERT OR UPDATE
  ON belegposabzu
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();
--

-- Quell-Zuschläge mit Typ 'E' werden erledigt gesetzt bei Übernahme aus Bestellung
 -- DROP FUNCTION IF EXISTS belegposabzu__a10_iud_srcabzu_done() CASCADE;
CREATE OR REPLACE FUNCTION belegposabzu__a_10_iud_srcabzu_done() RETURNS TRIGGER AS $$
 BEGIN

  IF (TG_OP IN ('INSERT','UPDATE')) THEN
    IF (new.belpaz_source_table='ldsabzu') AND (new.belpaz_type = 'E')  THEN
      PERFORM TWawi.Abzu_Set_Done(new.belpaz_source_table, new.belpaz_source_dbrid, TRUE, TRUE);
    END IF;
  ELSE
    IF (old.belpaz_source_table='ldsabzu') AND (old.belpaz_type = 'E')THEN
      -- TODO: Raise Notice beim wieder öffnen
      PERFORM TWawi.Abzu_Set_Done(old.belpaz_source_table, old.belpaz_source_dbrid, FALSE, TRUE);
   END IF;
  END IF;

  RETURN new;
 END $$ LANGUAGE plpgsql;
--
CREATE TRIGGER belegposabzu__a_10_iud_srcabzu_done
 AFTER INSERT OR UPDATE OR DELETE
 ON belegposabzu
 FOR EACH ROW
 EXECUTE PROCEDURE belegposabzu__a_10_iud_srcabzu_done();
--

CREATE OR REPLACE FUNCTION belegposabzu__a_iud() RETURNS TRIGGER AS $$
DECLARE docid INTEGER;
        beldokRec RECORD;
BEGIN

  /*
  IF BelPosAbzuTriggerOff() THEN -- Wir machen mal lieber nichts.

      IF Current_User='root' THEN  RAISE NOTICE 'belegposabzu__a_iud.Triggers=OFF. belegposabzu__a_iud belpaz_id: % belpaz_belegpos_id: %',
        --IfThen(tg_op = 'DELETE', old.belpaz_id, new.belpaz_id),
        --IfThen(tg_op = 'DELETE', old.belpaz_belegpos_id, new.belpaz_belegpos_id);
      END IF;

      RETURN new;
  END IF;

  IF Current_User='root' THEN RAISE NOTICE 'belegposabzu__a_iud belpaz_id: % belpaz_belegpos_id: %',
        --IfThen(tg_op = 'DELETE', old.belpaz_id, new.belpaz_id),
        --IfThen(tg_op = 'DELETE', old.belpaz_belegpos_id, new.belpaz_belegpos_id);

  END IF;*/

  -- Je nach dem was für eine Belegart das ist müssen wir noch dies oder das machen.
  IF (tg_op = 'DELETE') THEN
    SELECT belt_preiscalc, belp_dokument_id INTO beldokRec FROM belegpos JOIN belegtyp ON belp_belegtyp = belt_typ WHERE belp_id = old.belpaz_belegpos_id;

    --Das ist alles Preisberechnung, wird uebersprungen wenn Belegtyp keine Preise berechnet.
    IF NOT beldokRec.belt_preiscalc THEN
      RETURN old;
    END IF;

  ELSE
    SELECT belt_preiscalc, belp_dokument_id INTO beldokRec FROM belegpos JOIN belegtyp ON belp_belegtyp = belt_typ WHERE belp_id = new.belpaz_belegpos_id;
    --Das ist alles Preisberechnung, wird uebersprungen wenn Belegtyp keine Preise berechnet.
    IF NOT beldokRec.belt_preiscalc THEN
       RETURN new;
    END IF;
  END IF;


  --Hier kommen wir nur an, wenn der Beleg Preise berechnen soll.
  IF (tg_op = 'DELETE') THEN

    UPDATE belegpos SET belp_sumabzu = 0  WHERE belp_id = old.belpaz_belegpos_id;
    PERFORM TBeleg.Update_DokumentSumme(beldokRec.belp_dokument_id);
    RETURN old;
  ELSE

    IF (tg_op = 'INSERT') THEN
      PERFORM TBeleg.Update_DokumentSumme(beldokRec.belp_dokument_id);
      UPDATE belegpos SET belp_sumabzu = 0 WHERE belp_id = new.belpaz_belegpos_id;
    ELSE
      IF (
                (new.belpaz_betrag <> old.belpaz_betrag)
        OR      (new.belpaz_anzahl <> old.belpaz_anzahl)
        OR      (new.belpaz_steuproz <> old.belpaz_steuproz)
        OR      (new.belpaz_canskonto <> old.belpaz_canskonto)
        OR      (new.belpaz_canrabatt <> old.belpaz_canrabatt)
        ) THEN
        PERFORM TBeleg.Update_DokumentSumme(beldokRec.belp_dokument_id);
        UPDATE belegpos SET belp_sumabzu = 0 WHERE belp_id = new.belpaz_belegpos_id;
      END IF;
    END IF;
    RETURN new;
  END IF;

  --Meldung ausgeben,wennn Steuersatz ungültig
  IF ( (new.belpaz_belegtyp = 'ERG') AND (new.belpaz_steucode = (SELECT steu_z FROM steutxt WHERE steu_z=new.belpaz_steucode AND steu_valid_to<current_date)) ) THEN
        PERFORM PRODAT_TEXT(15810);
  END IF;


END $$ LANGUAGE plpgsql;

CREATE TRIGGER belegposabzu__a_iud
  AFTER INSERT OR UPDATE OR DELETE
  ON belegposabzu
  FOR EACH ROW
  EXECUTE PROCEDURE belegposabzu__a_iud();
---
--


-- Zahlungen auf einen Beleg (Eingangsrechnung) oder ein Dokument (Ausgangsrechnung) erfolgen.
CREATE TABLE belegzahlung
(
  belzlg_id                     SERIAL NOT NULL PRIMARY KEY,                        -- ID der Zahlung
  belzlg_dokument_id            INTEGER NOT NULL REFERENCES belegdokument_keys ON UPDATE CASCADE ON DELETE RESTRICT,  -- Belegdokument auf das die Zahlung erfolgt ist
  belzlg_tan                    VARCHAR(35),                                        -- Gibt es eine Transaktionsnummer oder sowas? Z.Bsp. Sepa_msgID
  belzlg_date                   DATE DEFAULT current_date,                          -- Zahlungsdatum
  belzlg_betrag                 NUMERIC(14,2) NOT NULL DEFAULT 0,                   -- Zahlbetrag
  belzlg_zahlart                INTEGER NOT NULL REFERENCES zahlart ON UPDATE CASCADE, -- Wie ist das Geld irgendwohin?!?
  belzlg_gebucht                BOOLEAN NOT NULL DEFAULT FALSE,                     -- Wenn gebucht, dann ist das erstmal erledigt
  belzlg_txt                    TEXT,                                               -- Für Anmerkungen und so Zeug.
  belzlg_txt_rtf                TEXT,
  -- belzlg_belzlg_id              INTEGER REFERENCES belegzahlung ON UPDATE CASCADE,  -- Verknüpfte Zahlung, z.Bsp. ID der Gutschrift die verrechnet wurde (ERG)
  --   Geht nicht wegen Vererbung: http://stackoverflow.com/questions/4940974/postgresql-foreign-key-not-existing-issue-of-inheritance
  --   Foreign key von eingrechzahlung auf belegzahlung macht sowas wie "SELECT belzlg_id FROM ONLY belegzahlung ... ", sieht also z.Bsp. andere eingrechzahlungen nicht.
  --   Workaround wäre eine Zwischentabelle zum Führen der Keys ...

  -- System (tables__generate_missing_fields)
  insert_date                   DATE,
  insert_by                     VARCHAR(32),
  modified_by                   VARCHAR(32),
  modified_date                 TIMESTAMP(0)
);


-- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
CREATE TRIGGER belegzahlung_set_modified
BEFORE INSERT OR UPDATE
ON belegzahlung
FOR EACH ROW
EXECUTE PROCEDURE table_modified();


CREATE OR REPLACE FUNCTION belegzahlung__a_iud() RETURNS TRIGGER AS $$
DECLARE zahlsumme  NUMERIC;
        dokumentid INTEGER;
BEGIN
  IF (tg_op = 'DELETE') THEN
    dokumentid:=old.belzlg_dokument_id;
  ELSE
    dokumentid:=new.belzlg_dokument_id;
  END IF;

  --Zahlungssumme in Dokument zurückschreiben
  UPDATE belegdokument SET beld_bezahlt = COALESCE( (SELECT SUM(belzlg_betrag) FROM belegzahlung WHERE belzlg_dokument_id = dokumentid AND belzlg_gebucht),0)
        WHERE beld_id = dokumentid;

  IF (tg_op = 'DELETE') THEN
    return old;
  ELSE
    return new;
  END IF;
END $$ LANGUAGE plpgsql;


CREATE TRIGGER belegzahlung__a_iud
AFTER INSERT OR UPDATE OR DELETE
ON belegzahlung
FOR EACH ROW
EXECUTE PROCEDURE belegzahlung__a_iud();



-- TCustomBelegModul

  -- Dokument wird erstellt/eingescannt -> mit Adresse und allen Artikeln verlinken
  CREATE OR REPLACE FUNCTION picndoku__a_60_iud__custombeleg() RETURNS TRIGGER AS $$
    DECLARE belegtyp VARCHAR;
    BEGIN
      IF TG_OP = 'UPDATE' THEN
        IF (new.pd_tablename IS NOT DISTINCT FROM old.pd_tablename) AND (new.pd_dbrid IS NOT DISTINCT FROM old.pd_dbrid) THEN
          RETURN new;
        END IF;
      END IF;

      IF TG_OP IN ('INSERT', 'UPDATE') THEN
        IF new.pd_tablename = 'belegdokument' THEN
          SELECT beld_belegtyp INTO belegtyp FROM belegdokument WHERE dbrid = new.pd_dbrid;
          IF TSystem.Settings__ENUM__Get('TCustomBelegModul.Art.DokuLinkTypes', belegtyp) THEN  --IF belegtyp IN TSystem.Settings__ENUM__Get('TCustomBelegModul.Art.DokuLinkTypes') THEN
            PERFORM CreateRecNoKeyword('art', belp_aknr, new.dbrid)
              FROM belegdokument
              JOIN belegpos ON belp_dokument_id = beld_id
              WHERE belegdokument.dbrid = new.pd_dbrid;
          END IF;
          IF TSystem.Settings__ENUM__Get('TCustomBelegModul.AdkL.DokuLinkTypes', belegtyp) THEN
            PERFORM CreateRecNoKeyword('adk', beld_krzlieferung, new.dbrid)
              FROM belegdokument
              WHERE belegdokument.dbrid = new.pd_dbrid;
          END IF;
        END IF;
      END IF;

      IF TG_OP IN ('UPDATE', 'DELETE') THEN
        IF old.pd_tablename = 'belegdokument' THEN
          SELECT beld_belegtyp INTO belegtyp FROM belegdokument WHERE dbrid = old.pd_dbrid;
          IF TSystem.Settings__ENUM__Get('TCustomBelegModul.Art.DokuLinkTypes', belegtyp) THEN
            PERFORM DeleteRecNoKeyword('art', belp_aknr, old.dbrid)
              FROM belegdokument
              JOIN belegpos ON belp_dokument_id = beld_id
              WHERE belegdokument.dbrid = old.pd_dbrid;
          END IF;
          IF TSystem.Settings__ENUM__Get('TCustomBelegModul.AdkL.DokuLinkTypes', belegtyp) THEN
            PERFORM DeleteRecNoKeyword('adk', beld_krzlieferung, old.dbrid)
              FROM belegdokument
              WHERE belegdokument.dbrid = old.pd_dbrid;
          END IF;
        END IF;
      END IF;

      IF TG_OP = 'DELETE' THEN
        RETURN old;
      ELSE
        RETURN new;
      END IF;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER picndoku__a_60_iud__custombeleg
      AFTER INSERT OR UPDATE OR DELETE
      ON picndoku
      FOR EACH ROW
      EXECUTE PROCEDURE picndoku__a_60_iud__custombeleg();
  --

  -- Belegdokument wird verändert -> Dokument mit richtiger Adresse verlinken
  CREATE OR REPLACE FUNCTION belegdokument__a_60_ud__custombeleg() RETURNS TRIGGER AS $$
    BEGIN
      IF TG_OP = 'UPDATE' THEN
        IF new.beld_krzlieferung IS NOT DISTINCT FROM old.beld_krzlieferung THEN
          RETURN new;
        END IF;
      END IF;

      IF TG_OP IN ('UPDATE') THEN  -- beim INSERT kann es noch keine angehängten Dokumente geben
        IF TSystem.Settings__ENUM__Get('TCustomBelegModul.AdkL.DokuLinkTypes', new.beld_belegtyp) THEN
          PERFORM CreateRecNoKeyword('adk', new.beld_krzlieferung, picndoku.dbrid)
            FROM picndoku
            WHERE pd_dbrid = new.dbrid;
        END IF;
      END IF;

      IF TG_OP IN ('UPDATE', 'DELETE') THEN
        IF TSystem.Settings__ENUM__Get('TCustomBelegModul.AdkL.DokuLinkTypes', old.beld_belegtyp) THEN
          PERFORM DeleteRecNoKeyword('adk', old.beld_krzlieferung, picndoku.dbrid)
            FROM picndoku
            WHERE pd_dbrid = old.dbrid;
        END IF;
      END IF;

      IF TG_OP = 'DELETE' THEN
        RETURN old;
      ELSE
        RETURN new;
      END IF;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belegdokument__a_60_ud__custombeleg
      AFTER UPDATE OR DELETE
      ON belegdokument
      FOR EACH ROW
      EXECUTE PROCEDURE belegdokument__a_60_ud__custombeleg();

  --- #18674 Stornolieferschein
  CREATE OR REPLACE FUNCTION belegdokument__a_90_u__beld_belegart__update() RETURNS TRIGGER AS $$
    DECLARE rec RECORD;

    BEGIN

      FOR rec IN SELECT belp_ag_id FROM belegpos WHERE belp_dokument_id = new.beld_id LOOP

          PERFORM TBeleg.LFS_Update_Auftg( rec.belp_ag_id );

      END LOOP;

      RETURN new;
   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belegdokument__a_90_u__beld_belegart__update
    AFTER UPDATE OF beld_belegart
    ON belegdokument
    FOR EACH ROW
   EXECUTE PROCEDURE belegdokument__a_90_u__beld_belegart__update();
---
  -- Belegposition wird verändert -> Dokumente mit Artikeln verlinken
  CREATE OR REPLACE FUNCTION belegpos__a_60_iud__custombeleg() RETURNS TRIGGER AS $$
    BEGIN
      IF TG_OP = 'UPDATE' THEN
        IF new.belp_aknr IS NOT DISTINCT FROM old.belp_aknr THEN
          RETURN new;
        END IF;
      END IF;

      IF TG_OP IN ('INSERT', 'UPDATE') THEN
        IF TSystem.Settings__ENUM__Get('TCustomBelegModul.Art.DokuLinkTypes', new.belp_belegtyp) THEN
          PERFORM CreateRecNoKeyword('art', new.belp_aknr, picndoku.dbrid)
            FROM belegdokument
            JOIN picndoku ON pd_dbrid = belegdokument.dbrid
            WHERE beld_id = new.belp_dokument_id;
        END IF;
      END IF;

      IF TG_OP IN ('UPDATE', 'DELETE') THEN
        IF TSystem.Settings__ENUM__Get('TCustomBelegModul.Art.DokuLinkTypes', old.belp_belegtyp) THEN
          PERFORM DeleteRecNoKeyword('art', old.belp_aknr, picndoku.dbrid)
            FROM belegdokument
            JOIN picndoku ON pd_dbrid = belegdokument.dbrid
            WHERE beld_id = old.belp_dokument_id;
        END IF;
      END IF;

      IF TG_OP = 'DELETE' THEN
        RETURN old;
      ELSE
        RETURN new;
      END IF;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belegpos__a_60_iud__custombeleg
      AFTER INSERT OR UPDATE OR DELETE
      ON belegpos
      FOR EACH ROW
      EXECUTE PROCEDURE belegpos__a_60_iud__custombeleg();
  --

--Lieferposition fertig fakturiert oder Fakt. aufgehoben.eingefuegt. Also Fakturiert-Status im Lieferdokument zuruecksetzen.
  CREATE OR REPLACE FUNCTION belegpos__a_80_iu__erledigt_storniert() RETURNS TRIGGER AS $$
    BEGIN
      IF BelPosTriggerOff() THEN -- Wir machen mal lieber nichts.
        RETURN new;
      END IF;

      --LIEFERSCHEIN
      IF (new.belp_belegtyp = 'LFS') THEN
        UPDATE belegdokument
          SET beld_verbucht = NOT EXISTS(SELECT true FROM belegpos WHERE belp_dokument_id = new.belp_dokument_id AND NOT belp_erledigt AND NOT belp_storniert)
        WHERE new.belp_dokument_id = beld_id;
      END IF;

      --Ursprungserklärung
      IF (new.belp_belegtyp = 'LUE') AND (TG_OP = 'UPDATE') THEN
        UPDATE belegdokument
          SET beld_definitiv = NOT EXISTS(SELECT true FROM belegpos WHERE belp_dokument_id = new.belp_dokument_id AND NOT belp_erledigt AND NOT belp_storniert)
        WHERE new.belp_dokument_id = beld_id;
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belegpos__a_80_iu__erledigt_storniert
      AFTER INSERT OR UPDATE
      OF belp_menge_done, belp_menge_done_gme, -- gelieferte Menge und
         belp_erledigt, belp_storniert -- Verechnungs-Flags
      ON belegpos
      FOR EACH ROW
      EXECUTE PROCEDURE belegpos__a_80_iu__erledigt_storniert();
  --

















